// ZIP Files Tool - Create ZIP archives from multiple files
let zipFiles = [];
let zipCreated = false; // Track if ZIP has been created

// Load JSZip library
function loadJSZip() {
    return new Promise((resolve, reject) => {
        if (window.JSZip) {
            resolve();
            return;
        }

        const script = document.createElement('script');
        script.src = 'https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js';
        script.async = true;
        script.onload = () => resolve();
        script.onerror = () => reject(new Error('Failed to load JSZip library'));
        document.head.appendChild(script);
    });
}

window.initializeZipFiles = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-file-archive"></i> Create ZIP Archive</h2>
                <p>Compress multiple files into a single ZIP archive</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="zip-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select files</h3>
                    <p>Upload multiple files to compress into ZIP</p>
                    <input type="file" id="zip-file-input" multiple style="display: none;">
                </div>
            </div>

            <div id="zip-file-list" class="zip-file-list hidden"></div>

            <div id="zip-options" class="hidden" style="margin: 1.5rem 0; padding: 1rem; background: var(--bg-light); border-radius: 8px; max-width: 600px; margin-left: auto; margin-right: auto;">
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Archive Name:</label>
                <input type="text" id="zip-name" class="form-input" placeholder="archive" value="archive" style="width: 100%; padding: 0.75rem; border: 1px solid var(--border-color); border-radius: 8px; font-size: 1rem;">
            </div>

            <div id="zip-actions" class="action-buttons hidden">
                <button onclick="clearZipFiles()" class="btn-secondary">
                    <i class="fas fa-trash-alt"></i> Clear All
                </button>
                <button onclick="createZipNow()" class="btn-primary">
                    <i class="fas fa-file-archive"></i> Create ZIP
                </button>
            </div>

            <div id="zip-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('zip-upload-zone').onclick = () => {
        document.getElementById('zip-file-input').click();
    };

    document.getElementById('zip-file-input').addEventListener('change', handleZipFileUpload);
    
    // Drag and drop
    const uploadZone = document.getElementById('zip-upload-zone');
    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.classList.add('dragover');
    });
    
    uploadZone.addEventListener('dragleave', () => {
        uploadZone.classList.remove('dragover');
    });
    
    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.classList.remove('dragover');
        const files = e.dataTransfer.files;
        handleZipFileUpload({ target: { files: files } });
    });
};

function handleZipFileUpload(e) {
    const files = Array.from(e.target.files);
    if (files.length === 0) return;

    // If ZIP was already created, restart the tool
    if (zipCreated) {
        zipFiles = [];
        zipCreated = false;
        document.getElementById('zip-result').classList.add('hidden');
        document.getElementById('zip-file-input').value = '';
    }

    zipFiles = zipFiles.concat(files);
    displayZipFileList();
}

function displayZipFileList() {
    const listDiv = document.getElementById('zip-file-list');
    const actionsDiv = document.getElementById('zip-actions');
    const optionsDiv = document.getElementById('zip-options');
    
    if (zipFiles.length === 0) {
        listDiv.classList.add('hidden');
        actionsDiv.classList.add('hidden');
        optionsDiv.classList.add('hidden');
        return;
    }
    
    listDiv.classList.remove('hidden');
    actionsDiv.classList.remove('hidden');
    optionsDiv.classList.remove('hidden');
    
    let totalSize = 0;
    let html = `<h3>📁 Selected Files (${zipFiles.length})</h3>`;
    html += `<div class="file-thumbnail-grid">`;
    
    zipFiles.forEach((file, index) => {
        const size = (file.size / 1024).toFixed(1);
        totalSize += file.size;
        
        const fileExtension = file.name.split('.').pop().toLowerCase();
        
        // Determine icon based on file type
        let icon = 'fa-file';
        if (['pdf'].includes(fileExtension)) icon = 'fa-file-pdf';
        else if (['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg'].includes(fileExtension)) icon = 'fa-file-image';
        else if (['doc', 'docx'].includes(fileExtension)) icon = 'fa-file-word';
        else if (['xls', 'xlsx'].includes(fileExtension)) icon = 'fa-file-excel';
        else if (['ppt', 'pptx'].includes(fileExtension)) icon = 'fa-file-powerpoint';
        else if (['zip', 'rar', '7z', 'tar', 'gz'].includes(fileExtension)) icon = 'fa-file-archive';
        else if (['mp3', 'wav', 'ogg', 'flac'].includes(fileExtension)) icon = 'fa-file-audio';
        else if (['mp4', 'avi', 'mov', 'mkv', 'webm'].includes(fileExtension)) icon = 'fa-file-video';
        else if (['txt', 'md', 'log'].includes(fileExtension)) icon = 'fa-file-alt';
        else if (['js', 'py', 'java', 'cpp', 'html', 'css', 'json', 'xml'].includes(fileExtension)) icon = 'fa-file-code';
        
        html += `
            <div class="file-thumbnail-item" data-file-index="${index}">
                <button onclick="removeZipFile(${index})" class="file-thumbnail-remove" title="Remove">
                    <i class="fas fa-times"></i>
                </button>
                <div class="file-thumbnail-preview">
                    <i class="fas ${icon} file-thumbnail-icon"></i>
                </div>
                <div class="file-thumbnail-name" title="${file.name}">${file.name}</div>
                <div class="file-thumbnail-info">${size} KB</div>
            </div>
        `;
    });

    const totalSizeMB = (totalSize / 1024 / 1024).toFixed(2);
    html += '</div>';
    html += `<div style="text-align: center; margin-top: 1rem; color: var(--text-secondary);">Total size: ${totalSizeMB} MB</div>`;
    listDiv.innerHTML = html;
    
    // Add image thumbnails for image files
    zipFiles.forEach((file, index) => {
        if (file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = function(e) {
                const thumbnailItem = listDiv.querySelector(`[data-file-index="${index}"] .file-thumbnail-preview`);
                if (thumbnailItem) {
                    thumbnailItem.innerHTML = `<img src="${e.target.result}" style="width: 100%; height: 100%; object-fit: cover; border-radius: 8px;" alt="${file.name}">`;
                }
            };
            reader.readAsDataURL(file);
        }
    });
}

function removeZipFile(index) {
    zipFiles.splice(index, 1);
    displayZipFileList();
}

function clearZipFiles() {
    zipFiles = [];
    zipCreated = false;
    displayZipFileList();
    document.getElementById('zip-result').classList.add('hidden');
    document.getElementById('zip-file-input').value = '';
}

async function createZipNow() {
    if (zipFiles.length === 0) {
        alert('Please select at least one file to compress');
        return;
    }
    
    const resultDiv = document.getElementById('zip-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Creating ZIP archive...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        await loadJSZip();

        const zip = new JSZip();
        
        // Add all files to ZIP
        for (const file of zipFiles) {
            zip.file(file.name, file);
        }
        
        // Generate ZIP file
        const blob = await zip.generateAsync({ 
            type: 'blob',
            compression: 'DEFLATE',
            compressionOptions: { level: 9 }
        });
        
        const url = URL.createObjectURL(blob);
        const zipName = document.getElementById('zip-name').value.trim() || 'archive';
        const fileName = zipName.endsWith('.zip') ? zipName : `${zipName}.zip`;
        const fileSize = (blob.size / 1024 / 1024).toFixed(2);
// Mark ZIP as created
        zipCreated = true;

        
        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ ZIP Archive Created Successfully!</h3>
                <p>${zipFiles.length} files compressed • ${fileSize} MB</p>
                <div style="margin-top: 1.5rem; text-align: center;">
                    <a href="${url}" download="${fileName}" class="btn-primary" style="text-decoration: none; display: inline-block; padding: 0.75rem 1.5rem; margin-right: 0.5rem;">
                        <i class="fas fa-download"></i> Download ${fileName}
                    </a>
                    <button onclick="clearZipFiles()" class="btn-secondary" style="padding: 0.75rem 1.5rem;">
                        <i class="fas fa-redo"></i> Create Another
                    </button>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('ZIP creation error:', error);
        resultDiv.innerHTML = `
            <div class="error-message">
                <i class="fas fa-exclamation-circle"></i>
                <h3>Error Creating ZIP</h3>
                <p>${error.message}</p>
            </div>
        `;
    }
}
