// YAML to JSON Converter

window.initializeYamlToJson = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-exchange-alt"></i> YAML to JSON Converter</h2>
                <p>Convert between YAML and JSON formats</p>
            </div>

            <div class="split-view">
                <div class="editor-section">
                    <h3>YAML Input</h3>
                    <textarea id="yaml-input" rows="15" class="form-input code-input" placeholder="name: John&#10;age: 30&#10;city: New York"></textarea>
                </div>

                <div class="editor-section">
                    <h3>JSON Output</h3>
                    <textarea id="json-output-yaml" rows="15" class="form-input code-input" readonly></textarea>
                </div>
            </div>

            <div class="button-group">
                <button onclick="convertYAMLtoJSON()" class="btn-primary">
                    <i class="fas fa-arrow-right"></i> YAML to JSON
                </button>
                <button onclick="convertJSONtoYAML()" class="btn-secondary">
                    <i class="fas fa-arrow-left"></i> JSON to YAML
                </button>
                <button onclick="copyYAMLJSON()" class="btn-download">
                    <i class="fas fa-copy"></i> Copy Output
                </button>
            </div>

            <div id="yaml-status" class="status-message"></div>
        </div>
    `;
};

function convertYAMLtoJSON() {
    const yamlInput = document.getElementById('yaml-input').value;
    const status = document.getElementById('yaml-status');

    try {
        const json = parseYAML(yamlInput);
        const formatted = JSON.stringify(json, null, 2);
        document.getElementById('json-output-yaml').value = formatted;
        status.innerHTML = '<div class="success">✅ Converted successfully!</div>';
        status.style.color = 'green';
    } catch (error) {
        status.innerHTML = '<div class="error">❌ Error: ' + error.message + '</div>';
        status.style.color = 'red';
    }
}

function convertJSONtoYAML() {
    const jsonInput = document.getElementById('json-output-yaml').value || document.getElementById('yaml-input').value;
    const status = document.getElementById('yaml-status');

    try {
        const obj = JSON.parse(jsonInput);
        const yaml = convertToYAML(obj);
        document.getElementById('yaml-input').value = yaml;
        status.innerHTML = '<div class="success">✅ Converted successfully!</div>';
        status.style.color = 'green';
    } catch (error) {
        status.innerHTML = '<div class="error">❌ Error: ' + error.message + '</div>';
        status.style.color = 'red';
    }
}

// Simple YAML parser (basic implementation)
function parseYAML(yaml) {
    const lines = yaml.split('\n').filter(line => line.trim() && !line.trim().startsWith('#'));
    const result = {};
    
    lines.forEach(line => {
        const match = line.match(/^(\s*)([^:]+):\s*(.*)$/);
        if (match) {
            const key = match[2].trim();
            let value = match[3].trim();
            
            // Parse value
            if (value === 'true') value = true;
            else if (value === 'false') value = false;
            else if (!isNaN(value) && value !== '') value = parseFloat(value);
            else if (value.startsWith('"') && value.endsWith('"')) value = value.slice(1, -1);
            else if (value.startsWith("'") && value.endsWith("'")) value = value.slice(1, -1);
            
            result[key] = value;
        }
    });
    
    return result;
}

function convertToYAML(obj, indent = 0) {
    let yaml = '';
    const spaces = '  '.repeat(indent);
    
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'object' && value !== null && !Array.isArray(value)) {
            yaml += `${spaces}${key}:\n${convertToYAML(value, indent + 1)}`;
        } else if (Array.isArray(value)) {
            yaml += `${spaces}${key}:\n`;
            value.forEach(item => {
                if (typeof item === 'object') {
                    yaml += `${spaces}  -\n${convertToYAML(item, indent + 2)}`;
                } else {
                    yaml += `${spaces}  - ${item}\n`;
                }
            });
        } else {
            const val = typeof value === 'string' ? `"${value}"` : value;
            yaml += `${spaces}${key}: ${val}\n`;
        }
    }
    
    return yaml;
}

function copyYAMLJSON() {
    const output = document.getElementById('json-output-yaml');
    if (!output.value) {
        alert('Nothing to copy');
        return;
    }
    output.select();
    document.execCommand('copy');
    alert('Copied to clipboard!');
}
