// Whiteboard / Drawing Tool

let whiteboardCanvas;
let whiteboardCtx;
let isWhiteboardDrawing = false;
let currentTool = 'pen';
let currentColor = '#000000';
let currentLineWidth = 2;

window.initializeWhiteboard = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-chalkboard"></i> Whiteboard</h2>
                <p>Draw, sketch, and annotate</p>
            </div>

            <div class="whiteboard-toolbar">
                <div class="toolbar-group">
                    <button onclick="selectTool('pen')" class="tool-btn active" id="pen-btn">
                        <i class="fas fa-pen"></i> Pen
                    </button>
                    <button onclick="selectTool('eraser')" class="tool-btn" id="eraser-btn">
                        <i class="fas fa-eraser"></i> Eraser
                    </button>
                    <button onclick="selectTool('line')" class="tool-btn" id="line-btn">
                        <i class="fas fa-minus"></i> Line
                    </button>
                    <button onclick="selectTool('rect')" class="tool-btn" id="rect-btn">
                        <i class="far fa-square"></i> Rectangle
                    </button>
                    <button onclick="selectTool('circle')" class="tool-btn" id="circle-btn">
                        <i class="far fa-circle"></i> Circle
                    </button>
                </div>

                <div class="toolbar-group">
                    <label>Color:</label>
                    <input type="color" id="wb-color" value="#000000" onchange="changeColor(this.value)">
                </div>

                <div class="toolbar-group">
                    <label>Size:</label>
                    <input type="range" id="wb-size" min="1" max="20" value="2" onchange="changeLineWidth(this.value)">
                    <span id="wb-size-value">2</span>px
                </div>

                <div class="toolbar-group">
                    <button onclick="clearWhiteboard()" class="btn-secondary">
                        <i class="fas fa-trash"></i> Clear
                    </button>
                    <button onclick="downloadWhiteboard()" class="btn-primary">
                        <i class="fas fa-download"></i> Save
                    </button>
                </div>
            </div>

            <canvas id="whiteboard-canvas" width="800" height="600"></canvas>
        </div>
    `;

    initializeWhiteboard();
};

function initializeWhiteboard() {
    whiteboardCanvas = document.getElementById('whiteboard-canvas');
    whiteboardCtx = whiteboardCanvas.getContext('2d');
    
    whiteboardCtx.fillStyle = '#ffffff';
    whiteboardCtx.fillRect(0, 0, whiteboardCanvas.width, whiteboardCanvas.height);
    
    whiteboardCtx.strokeStyle = currentColor;
    whiteboardCtx.lineWidth = currentLineWidth;
    whiteboardCtx.lineCap = 'round';
    whiteboardCtx.lineJoin = 'round';

    whiteboardCanvas.addEventListener('mousedown', startWhiteboardDrawing);
    whiteboardCanvas.addEventListener('mousemove', whiteboardDraw);
    whiteboardCanvas.addEventListener('mouseup', stopWhiteboardDrawing);
    whiteboardCanvas.addEventListener('mouseout', stopWhiteboardDrawing);

    whiteboardCanvas.addEventListener('touchstart', handleWhiteboardTouch);
    whiteboardCanvas.addEventListener('touchmove', handleWhiteboardTouch);
    whiteboardCanvas.addEventListener('touchend', stopWhiteboardDrawing);
}

let startX, startY;

function startWhiteboardDrawing(e) {
    isWhiteboardDrawing = true;
    const rect = whiteboardCanvas.getBoundingClientRect();
    startX = e.clientX - rect.left;
    startY = e.clientY - rect.top;
    
    if (currentTool === 'pen' || currentTool === 'eraser') {
        whiteboardCtx.beginPath();
        whiteboardCtx.moveTo(startX, startY);
    }
}

function whiteboardDraw(e) {
    if (!isWhiteboardDrawing) return;
    
    const rect = whiteboardCanvas.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;

    if (currentTool === 'pen') {
        whiteboardCtx.lineTo(x, y);
        whiteboardCtx.stroke();
    } else if (currentTool === 'eraser') {
        whiteboardCtx.clearRect(x - currentLineWidth / 2, y - currentLineWidth / 2, currentLineWidth, currentLineWidth);
    }
}

function stopWhiteboardDrawing(e) {
    if (!isWhiteboardDrawing) return;
    isWhiteboardDrawing = false;

    const rect = whiteboardCanvas.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;

    if (currentTool === 'line') {
        whiteboardCtx.beginPath();
        whiteboardCtx.moveTo(startX, startY);
        whiteboardCtx.lineTo(x, y);
        whiteboardCtx.stroke();
    } else if (currentTool === 'rect') {
        whiteboardCtx.strokeRect(startX, startY, x - startX, y - startY);
    } else if (currentTool === 'circle') {
        const radius = Math.sqrt(Math.pow(x - startX, 2) + Math.pow(y - startY, 2));
        whiteboardCtx.beginPath();
        whiteboardCtx.arc(startX, startY, radius, 0, 2 * Math.PI);
        whiteboardCtx.stroke();
    }
}

function handleWhiteboardTouch(e) {
    e.preventDefault();
    const touch = e.touches[0];
    const mouseEvent = new MouseEvent(e.type === 'touchstart' ? 'mousedown' : 'mousemove', {
        clientX: touch.clientX,
        clientY: touch.clientY
    });
    whiteboardCanvas.dispatchEvent(mouseEvent);
}

function selectTool(tool) {
    currentTool = tool;
    document.querySelectorAll('.tool-btn').forEach(btn => btn.classList.remove('active'));
    document.getElementById(tool + '-btn').classList.add('active');

    if (tool === 'eraser') {
        whiteboardCtx.globalCompositeOperation = 'destination-out';
    } else {
        whiteboardCtx.globalCompositeOperation = 'source-over';
        whiteboardCtx.strokeStyle = currentColor;
    }
}

function changeColor(color) {
    currentColor = color;
    if (currentTool !== 'eraser') {
        whiteboardCtx.strokeStyle = color;
    }
}

function changeLineWidth(width) {
    currentLineWidth = width;
    whiteboardCtx.lineWidth = width;
    document.getElementById('wb-size-value').textContent = width;
}

function clearWhiteboard() {
    whiteboardCtx.fillStyle = '#ffffff';
    whiteboardCtx.fillRect(0, 0, whiteboardCanvas.width, whiteboardCanvas.height);
}

function downloadWhiteboard() {
    const link = document.createElement('a');
    link.download = 'whiteboard.png';
    link.href = whiteboardCanvas.toDataURL();
    link.click();
}
