// Watermark PDF - Add text/image watermarks
let watermarkPdfFile = null;
let watermarkPdfDoc = null;

window.initializeWatermark = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-stamp"></i> Add Watermark</h2>
                <p>Add text watermark to your PDF</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="watermark-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <input type="file" id="watermark-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="watermark-options" class="split-options hidden">
                <h3>Watermark Settings</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 1.5rem;">
                    <div class="form-group">
                        <label>Watermark Text</label>
                        <input type="text" id="watermark-text" class="form-input" placeholder="CONFIDENTIAL" value="CONFIDENTIAL">
                    </div>
                    <div class="form-group">
                        <label>Font Size</label>
                        <input type="number" id="watermark-size" class="form-input" value="60" min="20" max="150">
                    </div>
                    <div class="form-group">
                        <label>Opacity</label>
                        <input type="range" id="watermark-opacity" class="form-input" min="0.1" max="1" step="0.1" value="0.3">
                        <span id="opacity-value">30%</span>
                    </div>
                    <div class="form-group">
                        <label>Rotation</label>
                        <select id="watermark-rotation" class="form-input">
                            <option value="0">Horizontal</option>
                            <option value="45" selected>Diagonal (45°)</option>
                            <option value="90">Vertical (90°)</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Color</label>
                        <input type="color" id="watermark-color" class="form-input" value="#ff0000">
                    </div>
                </div>
                
                <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                    <button class="btn-secondary" onclick="setWatermarkPreset('confidential')">CONFIDENTIAL</button>
                    <button class="btn-secondary" onclick="setWatermarkPreset('draft')">DRAFT</button>
                    <button class="btn-secondary" onclick="setWatermarkPreset('copy')">COPY</button>
                    <button class="btn-secondary" onclick="setWatermarkPreset('sample')">SAMPLE</button>
                </div>
            </div>

            <div id="watermark-actions" class="action-buttons hidden">
                <button onclick="addWatermarkNow()" class="btn-primary">
                    <i class="fas fa-stamp"></i> Add Watermark
                </button>
            </div>

            <div id="watermark-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('watermark-upload-zone').onclick = () => {
        document.getElementById('watermark-pdf-input').click();
    };

    document.getElementById('watermark-pdf-input').addEventListener('change', handleWatermarkPdfUpload);
    
    document.getElementById('watermark-opacity').addEventListener('input', (e) => {
        document.getElementById('opacity-value').textContent = Math.round(e.target.value * 100) + '%';
    });
};

async function handleWatermarkPdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        watermarkPdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        watermarkPdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        document.getElementById('watermark-options').classList.remove('hidden');
        document.getElementById('watermark-actions').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

function setWatermarkPreset(preset) {
    const presets = {
        confidential: { text: 'CONFIDENTIAL', color: '#ff0000', size: 60 },
        draft: { text: 'DRAFT', color: '#888888', size: 70 },
        copy: { text: 'COPY', color: '#0066ff', size: 60 },
        sample: { text: 'SAMPLE', color: '#ff6600', size: 60 }
    };
    
    const p = presets[preset];
    document.getElementById('watermark-text').value = p.text;
    document.getElementById('watermark-color').value = p.color;
    document.getElementById('watermark-size').value = p.size;
}

async function addWatermarkNow() {
    if (!watermarkPdfDoc) return;

    const text = document.getElementById('watermark-text').value;
    if (!text) {
        alert('Please enter watermark text');
        return;
    }

    const resultDiv = document.getElementById('watermark-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Adding Watermark...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        const fontSize = parseInt(document.getElementById('watermark-size').value);
        const opacity = parseFloat(document.getElementById('watermark-opacity').value);
        const rotation = parseInt(document.getElementById('watermark-rotation').value);
        const color = document.getElementById('watermark-color').value;
        
        const r = parseInt(color.substr(1, 2), 16) / 255;
        const g = parseInt(color.substr(3, 2), 16) / 255;
        const b = parseInt(color.substr(5, 2), 16) / 255;

        const pages = watermarkPdfDoc.getPages();

        for (const page of pages) {
            const { width, height } = page.getSize();
            const textWidth = text.length * fontSize * 0.5;
            
            let x = (width - textWidth) / 2;
            let y = height / 2;

            page.drawText(text, {
                x, y,
                size: fontSize,
                color: PDFLib.rgb(r, g, b),
                opacity: opacity,
                rotate: PDFLib.degrees(rotation)
            });
        }

        const pdfBytes = await watermarkPdfDoc.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ Watermark Added!</h3>
                <p>Watermark applied to ${pages.length} pages</p>
                <div class="button-group">
                    <a href="${url}" download="watermarked.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download PDF
                    </a>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Watermark error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
