// Watermark Adder Tool
function initializeWatermarkTool() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="watermark-container">
            <h2>💧 Watermark Tool</h2>
            <p class="tool-description">Add text or image watermarks to your photos</p>

            <div class="tabs">
                <button class="tab-btn active" onclick="switchWatermarkTab('text')">Text Watermark</button>
                <button class="tab-btn" onclick="switchWatermarkTab('image')">Image Watermark</button>
            </div>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="watermark-images" accept="image/*" multiple onchange="loadWatermarkImages(this)">
                    <span>📁 Choose Images (Multiple)</span>
                </label>
            </div>

            <div id="watermark-text-tab" class="tab-content active">
                <h3>Text Watermark Settings</h3>
                
                <div class="form-group">
                    <label>Watermark Text:</label>
                    <input type="text" id="watermark-text" placeholder="© Your Name 2025" value="© Copyright 2025">
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Font Size: <span id="wm-fontsize-val">36</span>px</label>
                        <input type="range" id="wm-fontsize" min="12" max="96" value="36" oninput="updateWMFontSize()">
                    </div>
                    <div class="form-group">
                        <label>Opacity: <span id="wm-opacity-val">50</span>%</label>
                        <input type="range" id="wm-opacity" min="0" max="100" value="50" oninput="updateWMOpacity()">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Text Color:</label>
                        <input type="color" id="wm-color" value="#ffffff">
                    </div>
                    <div class="form-group">
                        <label>Position:</label>
                        <select id="wm-position">
                            <option value="bottom-right">Bottom Right</option>
                            <option value="bottom-left">Bottom Left</option>
                            <option value="top-right">Top Right</option>
                            <option value="top-left">Top Left</option>
                            <option value="center">Center</option>
                            <option value="tile">Tile (Repeat)</option>
                        </select>
                    </div>
                </div>
            </div>

            <div id="watermark-image-tab" class="tab-content">
                <h3>Image Watermark Settings</h3>
                
                <div class="form-group">
                    <label>Watermark Image:</label>
                    <input type="file" id="watermark-logo" accept="image/*">
                </div>

                <div class="form-group">
                    <label>Size: <span id="wm-size-val">20</span>%</label>
                    <input type="range" id="wm-size" min="5" max="50" value="20" oninput="updateWMSize()">
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Opacity: <span id="wm-img-opacity-val">70</span>%</label>
                        <input type="range" id="wm-img-opacity" min="0" max="100" value="70" oninput="updateWMImgOpacity()">
                    </div>
                    <div class="form-group">
                        <label>Position:</label>
                        <select id="wm-img-position">
                            <option value="bottom-right">Bottom Right</option>
                            <option value="bottom-left">Bottom Left</option>
                            <option value="top-right">Top Right</option>
                            <option value="top-left">Top Left</option>
                            <option value="center">Center</option>
                        </select>
                    </div>
                </div>
            </div>

            <button class="btn btn-primary" onclick="applyWatermark()">Apply Watermark</button>

            <div id="watermark-results" class="result-section hidden">
                <h3>Watermarked Images</h3>
                <div id="watermark-grid" class="results-grid"></div>
                <button class="btn btn-secondary" onclick="downloadAllWatermarked()">📦 Download All</button>
            </div>
        </div>
    `;
}

let watermarkSourceImages = [];

function switchWatermarkTab(tab) {
    const tabs = document.querySelectorAll('.tab-btn');
    const contents = document.querySelectorAll('.tab-content');
    
    tabs.forEach(t => t.classList.remove('active'));
    contents.forEach(c => c.classList.remove('active'));
    
    if (tab === 'text') {
        tabs[0].classList.add('active');
        document.getElementById('watermark-text-tab').classList.add('active');
    } else {
        tabs[1].classList.add('active');
        document.getElementById('watermark-image-tab').classList.add('active');
    }
}

function loadWatermarkImages(input) {
    watermarkSourceImages = Array.from(input.files);
}

function updateWMFontSize() {
    document.getElementById('wm-fontsize-val').textContent = 
        document.getElementById('wm-fontsize').value;
}

function updateWMOpacity() {
    document.getElementById('wm-opacity-val').textContent = 
        document.getElementById('wm-opacity').value;
}

function updateWMSize() {
    document.getElementById('wm-size-val').textContent = 
        document.getElementById('wm-size').value;
}

function updateWMImgOpacity() {
    document.getElementById('wm-img-opacity-val').textContent = 
        document.getElementById('wm-img-opacity').value;
}

async function applyWatermark() {
    if (watermarkSourceImages.length === 0) {
        alert('Please select images first');
        return;
    }

    const formData = new FormData();
    watermarkSourceImages.forEach(img => formData.append('images', img));

    const activeTab = document.getElementById('watermark-text-tab').classList.contains('active') ? 'text' : 'image';
    formData.append('type', activeTab);

    if (activeTab === 'text') {
        formData.append('text', document.getElementById('watermark-text').value);
        formData.append('fontSize', document.getElementById('wm-fontsize').value);
        formData.append('color', document.getElementById('wm-color').value);
        formData.append('opacity', document.getElementById('wm-opacity').value);
        formData.append('position', document.getElementById('wm-position').value);
    } else {
        const logo = document.getElementById('watermark-logo').files[0];
        if (!logo) {
            alert('Please select a watermark image');
            return;
        }
        formData.append('logo', logo);
        formData.append('size', document.getElementById('wm-size').value);
        formData.append('opacity', document.getElementById('wm-img-opacity').value);
        formData.append('position', document.getElementById('wm-img-position').value);
    }

    try {
        const response = await fetch('/api/image/add-watermark', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayWatermarkedImages(data.images);
        } else {
            alert('Failed to apply watermark: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while applying watermark');
    }
}

function displayWatermarkedImages(images) {
    const grid = document.getElementById('watermark-grid');
    grid.innerHTML = '';

    images.forEach(img => {
        const item = document.createElement('div');
        item.className = 'result-item';
        item.innerHTML = `
            <img src="${img.url}" alt="${img.filename}">
            <button class="btn-small" onclick="downloadFile('${img.url}', '${img.filename}')">Download</button>
        `;
        grid.appendChild(item);
    });

    document.getElementById('watermark-results').classList.remove('hidden');
}

function downloadAllWatermarked() {
    alert('Downloading all watermarked images...');
}


// Expose to window for router
window.initializeWatermarkTool = initializeWatermarkTool;
