// Video Trimmer

let videoFile = null;

window.initializeVideoTrimmer = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-cut"></i> Video Trimmer</h2>
                <p>Trim and cut video clips</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('video-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop video here or click to upload</h3>
                <p>Supports: MP4, WebM, AVI, MOV</p>
                <input type="file" id="video-input" accept="video/*" style="display: none;">
            </div>

            <div id="video-preview" class="preview-section hidden">
                <video id="video-player" controls style="max-width: 100%;"></video>
                <div class="video-info">
                    <p><strong>Duration:</strong> <span id="video-duration">0:00</span></p>
                </div>
            </div>

            <div id="video-trim-options" class="options-section hidden">
                <h3>Trim Options</h3>
                <div class="form-group">
                    <label>Start Time (seconds):</label>
                    <input type="number" id="trim-start" min="0" value="0" step="0.1" class="form-input">
                    <button onclick="setCurrentTime('start')" class="btn-secondary">Use Current Time</button>
                </div>
                <div class="form-group">
                    <label>End Time (seconds):</label>
                    <input type="number" id="trim-end" min="0" value="10" step="0.1" class="form-input">
                    <button onclick="setCurrentTime('end')" class="btn-secondary">Use Current Time</button>
                </div>
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="trim-format">
                        <option value="mp4">MP4</option>
                        <option value="webm">WebM</option>
                        <option value="avi">AVI</option>
                    </select>
                </div>
                <button onclick="trimVideo()" class="btn-primary">
                    <i class="fas fa-cut"></i> Trim Video
                </button>
            </div>

            <div id="video-trim-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('video-input').addEventListener('change', handleVideoUpload);
};

function handleVideoUpload(e) {
    videoFile = e.target.files[0];
    if (!videoFile) return;

    const video = document.getElementById('video-player');
    video.src = URL.createObjectURL(videoFile);

    document.getElementById('video-preview').classList.remove('hidden');
    document.getElementById('video-trim-options').classList.remove('hidden');

    video.addEventListener('loadedmetadata', () => {
        const duration = video.duration;
        document.getElementById('video-duration').textContent = formatTime(duration);
        document.getElementById('trim-end').value = duration.toFixed(1);
        document.getElementById('trim-end').max = duration;
        document.getElementById('trim-start').max = duration;
    });
}

function setCurrentTime(type) {
    const video = document.getElementById('video-player');
    const currentTime = video.currentTime;
    
    if (type === 'start') {
        document.getElementById('trim-start').value = currentTime.toFixed(1);
    } else {
        document.getElementById('trim-end').value = currentTime.toFixed(1);
    }
}

function formatTime(seconds) {
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
}

async function trimVideo() {
    if (!videoFile) return;

    const startTime = parseFloat(document.getElementById('trim-start').value);
    const endTime = parseFloat(document.getElementById('trim-end').value);

    if (startTime >= endTime) {
        alert('End time must be greater than start time');
        return;
    }

    const resultsDiv = document.getElementById('video-trim-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Trimming video... This may take a while.</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('video', videoFile);
    formData.append('start', startTime);
    formData.append('end', endTime);
    formData.append('format', document.getElementById('trim-format').value);

    try {
        const response = await fetch('/api/video/trim', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Video Trimmed Successfully!</h3>
                    <p><strong>Duration:</strong> ${formatTime(endTime - startTime)}</p>
                    <p><strong>Size:</strong> ${(data.size / (1024 * 1024)).toFixed(2)} MB</p>
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download Video
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
