// Video to GIF Converter

let videoFile = null;

window.initializeVideoToGIF = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="gif-converter-interface">
            <div class="upload-section">
                <label class="upload-label">
                    <input type="file" id="gif-video-file" accept="video/*" onchange="loadVideoForGIF(this)" style="display: none;">
                    <div class="upload-box">
                        <i class="fas fa-video"></i>
                        <p>Click to select video or drag & drop</p>
                        <span>Supported: MP4, WebM, MOV, AVI</span>
                    </div>
                </label>
            </div>

            <div id="gif-preview" class="preview-section hidden">
                <h3>Preview</h3>
                <video id="gif-preview-video" controls></video>
            </div>

            <div id="gif-options" class="options-section hidden">
                <h3>GIF Settings</h3>
                
                <div class="form-group">
                    <label>Start Time (seconds):</label>
                    <input type="number" id="gif-start" value="0" min="0" step="0.1">
                </div>

                <div class="form-group">
                    <label>Duration (seconds): <span id="gif-end-time">5.0</span></label>
                    <input type="range" id="gif-duration" min="0.5" max="10" step="0.1" value="5" oninput="updateGIFDuration()">
                </div>

                <div class="form-group">
                    <label>Frame Rate (FPS):</label>
                    <select id="gif-fps">
                        <option value="10">10 FPS (Smaller)</option>
                        <option value="15" selected>15 FPS (Balanced)</option>
                        <option value="20">20 FPS (Smooth)</option>
                        <option value="30">30 FPS (Best Quality)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Width (pixels):</label>
                    <input type="number" id="gif-width" value="480" min="100" max="1920" step="10">
                </div>

                <div class="form-group">
                    <label>Quality:</label>
                    <div class="quality-buttons">
                        <button class="gif-quality-btn" onclick="selectGIFQuality('low')">Low</button>
                        <button class="gif-quality-btn active" onclick="selectGIFQuality('medium')">Medium</button>
                        <button class="gif-quality-btn" onclick="selectGIFQuality('high')">High</button>
                    </div>
                    <input type="hidden" id="gif-quality" value="medium">
                </div>

                <button onclick="convertToGIF()" class="btn-primary">🎬 Convert to GIF</button>
            </div>

            <div id="gif-result" class="result-section hidden"></div>
        </div>
    `;
};

function loadVideoForGIF(input) {
    videoFile = input.files[0];
    if (!videoFile) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        const video = document.getElementById('gif-preview-video');
        video.src = e.target.result;
        video.load();
        
        video.onloadedmetadata = function() {
            document.getElementById('gif-duration').max = video.duration;
            document.getElementById('gif-duration').value = Math.min(5, video.duration);
            document.getElementById('gif-end-time').textContent = Math.min(5, video.duration).toFixed(1);
            updateGIFDuration();
        };
        
        document.getElementById('gif-preview').classList.remove('hidden');
        document.getElementById('gif-options').classList.remove('hidden');
    };
    reader.readAsDataURL(videoFile);
}

function updateGIFDuration() {
    const duration = document.getElementById('gif-duration').value;
    document.getElementById('gif-end-time').textContent = parseFloat(duration).toFixed(1);
}

function selectGIFQuality(quality) {
    document.querySelectorAll('.gif-quality-btn').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    document.getElementById('gif-quality').value = quality;
}

async function convertToGIF() {
    if (!videoFile) return;

    const startTime = parseFloat(document.getElementById('gif-start').value);
    const duration = parseFloat(document.getElementById('gif-duration').value);
    const fps = parseInt(document.getElementById('gif-fps').value);
    const width = parseInt(document.getElementById('gif-width').value);
    const quality = document.getElementById('gif-quality').value;

    const formData = new FormData();
    formData.append('video', videoFile);
    formData.append('startTime', startTime);
    formData.append('duration', duration);
    formData.append('fps', fps);
    formData.append('width', width);
    formData.append('quality', quality);

    document.getElementById('gif-result').innerHTML = '<div class="processing">Converting video to GIF... This may take a moment...</div>';
    document.getElementById('gif-result').classList.remove('hidden');

    try {
        const response = await fetch('/api/creative/video-to-gif', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayGIFResult(data);
        } else {
            document.getElementById('gif-result').innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('gif-result').innerHTML = `<div class="error">Error converting video: ${error.message}</div>`;
    }
}

function displayGIFResult(data) {
    const resultHTML = `
        <div class="result-success">
            <h3>✅ GIF Created Successfully!</h3>
            <div class="gif-preview-result">
                <img src="${data.url}" alt="Generated GIF">
            </div>
            <div class="file-info">
                <p><strong>File Size:</strong> ${(data.size / 1024).toFixed(2)} KB</p>
                <p><strong>Dimensions:</strong> ${data.width}x${data.height}</p>
                <p><strong>Frames:</strong> ${data.frames}</p>
            </div>
            <button onclick="window.open('${data.url}', '_blank')" class="download-btn">
                📥 Download GIF
            </button>
        </div>
    `;
    document.getElementById('gif-result').innerHTML = resultHTML;
}
