// Video Thumbnail Generator

let videoFile = null;
let videoElement = null;

window.initializeVideoThumbnail = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-video"></i> Video Thumbnail Generator</h2>
                <p>Extract thumbnails from videos</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('video-thumb-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop video here or click to upload</h3>
                <p>Supports: MP4, WebM, AVI, MOV</p>
                <input type="file" id="video-thumb-input" accept="video/*" style="display: none;">
            </div>

            <div id="video-preview-section" class="hidden">
                <h3>Video Preview</h3>
                <video id="video-element" controls style="max-width: 100%; height: auto;"></video>

                <div class="thumbnail-controls">
                    <div class="form-group">
                        <label>Time Position (seconds):</label>
                        <input type="range" id="thumb-time" min="0" max="100" value="0" step="0.1" oninput="seekToTime()">
                        <span id="time-display">0.0s</span>
                    </div>

                    <div class="form-group">
                        <label>Thumbnail Size:</label>
                        <select id="thumb-size">
                            <option value="original">Original Size</option>
                            <option value="1920x1080">1920x1080 (Full HD)</option>
                            <option value="1280x720" selected>1280x720 (HD)</option>
                            <option value="854x480">854x480 (SD)</option>
                            <option value="640x360">640x360 (Small)</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Format:</label>
                        <select id="thumb-format">
                            <option value="jpeg">JPEG</option>
                            <option value="png">PNG</option>
                            <option value="webp">WebP</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Quality (JPEG/WebP only):</label>
                        <input type="range" id="thumb-quality" min="0" max="100" value="90">
                        <span id="quality-display">90%</span>
                    </div>

                    <button onclick="captureMultipleThumbnails()" class="btn-secondary">
                        <i class="fas fa-images"></i> Capture Multiple (Every 10s)
                    </button>
                    
                    <button onclick="captureThumbnail()" class="btn-primary">
                        <i class="fas fa-camera"></i> Capture Thumbnail
                    </button>
                </div>

                <div id="thumbnail-gallery" class="thumbnail-gallery hidden"></div>
            </div>
        </div>
    `;

    document.getElementById('video-thumb-input').addEventListener('change', handleVideoUpload);
    document.getElementById('thumb-quality').addEventListener('input', (e) => {
        document.getElementById('quality-display').textContent = e.target.value + '%';
    });
};

function handleVideoUpload(e) {
    videoFile = e.target.files[0];
    if (!videoFile) return;

    videoElement = document.getElementById('video-element');
    videoElement.src = URL.createObjectURL(videoFile);

    videoElement.addEventListener('loadedmetadata', () => {
        const duration = videoElement.duration;
        document.getElementById('thumb-time').max = duration;
        document.getElementById('time-display').textContent = '0.0s';
    });

    document.getElementById('video-preview-section').classList.remove('hidden');
}

function seekToTime() {
    const time = parseFloat(document.getElementById('thumb-time').value);
    document.getElementById('time-display').textContent = time.toFixed(1) + 's';
    videoElement.currentTime = time;
}

function captureThumbnail() {
    if (!videoElement) return;

    const canvas = document.createElement('canvas');
    const sizeOption = document.getElementById('thumb-size').value;
    
    if (sizeOption === 'original') {
        canvas.width = videoElement.videoWidth;
        canvas.height = videoElement.videoHeight;
    } else {
        const [width, height] = sizeOption.split('x').map(Number);
        canvas.width = width;
        canvas.height = height;
    }

    const ctx = canvas.getContext('2d');
    ctx.drawImage(videoElement, 0, 0, canvas.width, canvas.height);

    const format = document.getElementById('thumb-format').value;
    const quality = parseInt(document.getElementById('thumb-quality').value) / 100;
    
    let dataUrl;
    if (format === 'jpeg') {
        dataUrl = canvas.toDataURL('image/jpeg', quality);
    } else if (format === 'png') {
        dataUrl = canvas.toDataURL('image/png');
    } else if (format === 'webp') {
        dataUrl = canvas.toDataURL('image/webp', quality);
    }

    // Download
    const link = document.createElement('a');
    const time = videoElement.currentTime.toFixed(1);
    link.download = `thumbnail-${time}s.${format === 'jpeg' ? 'jpg' : format}`;
    link.href = dataUrl;
    link.click();

    // Show in gallery
    showThumbnailInGallery(dataUrl, time);
}

function captureMultipleThumbnails() {
    if (!videoElement) return;

    const duration = videoElement.duration;
    const interval = 10; // Capture every 10 seconds
    const gallery = document.getElementById('thumbnail-gallery');
    gallery.innerHTML = '<h3>Captured Thumbnails</h3>';
    gallery.classList.remove('hidden');

    let currentTime = 0;
    const captureNext = () => {
        if (currentTime > duration) return;

        videoElement.currentTime = currentTime;
        
        videoElement.addEventListener('seeked', function captureCurrent() {
            const canvas = document.createElement('canvas');
            canvas.width = 320;
            canvas.height = 180;
            const ctx = canvas.getContext('2d');
            ctx.drawImage(videoElement, 0, 0, canvas.width, canvas.height);
            
            showThumbnailInGallery(canvas.toDataURL('image/jpeg', 0.9), currentTime.toFixed(1));
            
            videoElement.removeEventListener('seeked', captureCurrent);
            currentTime += interval;
            setTimeout(captureNext, 100);
        }, { once: true });
    };

    captureNext();
}

function showThumbnailInGallery(dataUrl, time) {
    const gallery = document.getElementById('thumbnail-gallery');
    gallery.classList.remove('hidden');
    
    const thumbDiv = document.createElement('div');
    thumbDiv.className = 'thumbnail-item';
    thumbDiv.innerHTML = `
        <img src="${dataUrl}" alt="Thumbnail at ${time}s">
        <p>${time}s</p>
    `;
    gallery.appendChild(thumbDiv);
}
