// Video Compressor

let videoToCompress = null;

window.initializeVideoCompressor = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-compress"></i> Video Compressor</h2>
                <p>Compress video files to reduce size</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('video-compress-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop video file here or click to upload</h3>
                <p>Supports: MP4, AVI, MOV, MKV, WebM</p>
                <input type="file" id="video-compress-input" accept="video/*" style="display: none;">
            </div>

            <div id="video-compress-preview" class="preview-section hidden">
                <h3>Video Preview</h3>
                <video id="compress-video-player" controls style="max-width: 100%; max-height: 400px;"></video>
                <div id="compress-video-info"></div>
            </div>

            <div id="video-compress-options" class="options-section hidden">
                <h3>Compression Options</h3>
                
                <div class="form-group">
                    <label>Quality Preset:</label>
                    <select id="compress-quality">
                        <option value="high">High Quality (Less Compression)</option>
                        <option value="medium" selected>Medium Quality (Balanced)</option>
                        <option value="low">Low Quality (Maximum Compression)</option>
                        <option value="custom">Custom</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Resolution:</label>
                    <select id="compress-resolution">
                        <option value="original">Keep Original</option>
                        <option value="1080">1080p (1920x1080)</option>
                        <option value="720">720p (1280x720)</option>
                        <option value="480">480p (854x480)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Target Size (MB - optional):</label>
                    <input type="number" id="compress-target-size" class="form-input" placeholder="Leave empty for default compression">
                </div>

                <button onclick="compressVideo()" class="btn-primary">
                    <i class="fas fa-compress"></i> Compress Video
                </button>
            </div>

            <div id="video-compress-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('video-compress-input').addEventListener('change', handleVideoCompressUpload);
};

function handleVideoCompressUpload(e) {
    videoToCompress = e.target.files[0];
    if (!videoToCompress) return;

    const player = document.getElementById('compress-video-player');
    player.src = URL.createObjectURL(videoToCompress);

    document.getElementById('compress-video-info').innerHTML = `
        <p><strong>File:</strong> ${videoToCompress.name}</p>
        <p><strong>Size:</strong> ${(videoToCompress.size / 1024 / 1024).toFixed(2)} MB</p>
        <p><strong>Type:</strong> ${videoToCompress.type}</p>
    `;

    document.getElementById('video-compress-preview').classList.remove('hidden');
    document.getElementById('video-compress-options').classList.remove('hidden');
}

async function compressVideo() {
    if (!videoToCompress) return;

    const resultsDiv = document.getElementById('video-compress-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Compressing video... This may take a while.</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('video', videoToCompress);
    formData.append('quality', document.getElementById('compress-quality').value);
    formData.append('resolution', document.getElementById('compress-resolution').value);
    formData.append('targetSize', document.getElementById('compress-target-size').value || '');

    try {
        const response = await fetch('/api/video/compress', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            const originalSize = (videoToCompress.size / 1024 / 1024).toFixed(2);
            const compressedSize = data.size ? (data.size / 1024 / 1024).toFixed(2) : 'N/A';
            const reduction = data.size ? (((videoToCompress.size - data.size) / videoToCompress.size) * 100).toFixed(1) : 'N/A';

            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Video Compressed!</h3>
                    <div class="compression-stats">
                        <p><strong>Original Size:</strong> ${originalSize} MB</p>
                        <p><strong>Compressed Size:</strong> ${compressedSize} MB</p>
                        <p><strong>Size Reduction:</strong> ${reduction}%</p>
                    </div>
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download Compressed Video
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
