// UUID Generator

window.initializeUuidGenerator = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-fingerprint"></i> UUID Generator</h2>
                <p>Generate unique identifiers (UUIDs/GUIDs)</p>
            </div>

            <div class="form-container">
                <div class="form-group">
                    <label>UUID Version:</label>
                    <select id="uuid-version">
                        <option value="v4">Version 4 (Random)</option>
                        <option value="v1">Version 1 (Timestamp-based)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Number of UUIDs:</label>
                    <input type="number" id="uuid-count" min="1" max="100" value="1" class="form-input">
                </div>

                <div class="form-group">
                    <label>Format:</label>
                    <select id="uuid-format">
                        <option value="default">Standard (with hyphens)</option>
                        <option value="compact">Compact (no hyphens)</option>
                        <option value="uppercase">Uppercase</option>
                        <option value="braces">With braces {...}</option>
                    </select>
                </div>

                <button onclick="generateUUIDs()" class="btn-primary">
                    <i class="fas fa-random"></i> Generate UUIDs
                </button>

                <div id="uuid-results" class="results-section hidden">
                    <h3>Generated UUIDs:</h3>
                    <textarea id="uuid-output" rows="10" class="form-input code-input" readonly></textarea>
                    <button onclick="copyUUIDs()" class="btn-download">
                        <i class="fas fa-copy"></i> Copy All
                    </button>
                </div>
            </div>

            <div class="info-box">
                <h4>ℹ️ UUID Information</h4>
                <p><strong>Version 4:</strong> Randomly generated, most commonly used</p>
                <p><strong>Version 1:</strong> Based on timestamp and MAC address</p>
                <p><strong>Format:</strong> xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx</p>
            </div>
        </div>
    `;
};

function generateUUIDs() {
    const version = document.getElementById('uuid-version').value;
    const count = parseInt(document.getElementById('uuid-count').value);
    const format = document.getElementById('uuid-format').value;

    let uuids = [];

    for (let i = 0; i < count; i++) {
        let uuid = version === 'v4' ? generateUUIDv4() : generateUUIDv1();
        uuid = formatUUID(uuid, format);
        uuids.push(uuid);
    }

    document.getElementById('uuid-output').value = uuids.join('\n');
    document.getElementById('uuid-results').classList.remove('hidden');
}

function generateUUIDv4() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
        const r = Math.random() * 16 | 0;
        const v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
    });
}

function generateUUIDv1() {
    // Simplified v1 implementation
    const now = Date.now();
    const time = now * 10000 + 122192928000000000;
    
    const timeLow = (time & 0xffffffff).toString(16).padStart(8, '0');
    const timeMid = ((time / 0x100000000) & 0xffff).toString(16).padStart(4, '0');
    const timeHi = (((time / 0x1000000000000) & 0x0fff) | 0x1000).toString(16).padStart(4, '0');
    const clockSeq = Math.floor(Math.random() * 0x3fff).toString(16).padStart(4, '0');
    const node = Math.floor(Math.random() * 0xffffffffffff).toString(16).padStart(12, '0');

    return `${timeLow}-${timeMid}-${timeHi}-${clockSeq}-${node}`;
}

function formatUUID(uuid, format) {
    switch(format) {
        case 'compact':
            return uuid.replace(/-/g, '');
        case 'uppercase':
            return uuid.toUpperCase();
        case 'braces':
            return `{${uuid}}`;
        default:
            return uuid;
    }
}

function copyUUIDs() {
    const output = document.getElementById('uuid-output');
    output.select();
    document.execCommand('copy');
    alert('UUIDs copied to clipboard!');
}
