// URL Shortener & QR Code Tool
function initializeURLShortener() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="url-shortener-container">
            <h2>🔗 URL Shortener & Manager</h2>
            <p class="tool-description">Create short links with custom aliases and QR codes</p>

            <div class="url-form">
                <div class="form-group">
                    <label>Long URL:</label>
                    <input type="url" id="long-url" placeholder="https://example.com/very/long/url" required>
                </div>

                <div class="form-group">
                    <label>Custom Alias (Optional):</label>
                    <input type="text" id="custom-alias" placeholder="my-link" pattern="[a-zA-Z0-9-]+">
                    <small>Leave empty for random alias</small>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="generate-qr" checked>
                        Generate QR Code
                    </label>
                </div>

                <button class="btn btn-primary" onclick="shortenURL()">Shorten URL</button>
            </div>

            <div id="url-result" class="result-section hidden">
                <h3>Your Short Link</h3>
                <div class="short-url-display">
                    <input type="text" id="short-url-input" readonly>
                    <button class="btn btn-secondary" onclick="copyShortURL()">📋 Copy</button>
                </div>

                <div id="qr-code-display" class="qr-display hidden">
                    <h4>QR Code</h4>
                    <img id="qr-code-img" src="" alt="QR Code">
                    <button class="btn btn-secondary" onclick="downloadQR()">Download QR</button>
                </div>

                <div class="url-stats">
                    <h4>Link Statistics</h4>
                    <div id="url-stats-display"></div>
                </div>
            </div>

            <div class="url-history">
                <h3>Recent Links</h3>
                <div id="url-history-list"></div>
            </div>
        </div>
    `;
    
    loadURLHistory();
}

let urlHistory = [];

function loadURLHistory() {
    const saved = localStorage.getItem('urlHistory');
    if (saved) {
        urlHistory = JSON.parse(saved);
        displayURLHistory();
    }
}

function saveURLHistory() {
    localStorage.setItem('urlHistory', JSON.stringify(urlHistory.slice(0, 10)));
}

async function shortenURL() {
    const longURL = document.getElementById('long-url').value;
    const customAlias = document.getElementById('custom-alias').value;
    const generateQR = document.getElementById('generate-qr').checked;

    if (!longURL) {
        alert('Please enter a URL');
        return;
    }

    try {
        const response = await fetch('/api/utility/shorten-url', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ url: longURL, alias: customAlias, generateQR })
        });

        const data = await response.json();

        if (data.success) {
            displayShortURL(data);
            addToHistory(data);
        } else {
            alert('Failed to shorten URL: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while shortening URL');
    }
}

function displayShortURL(data) {
    document.getElementById('short-url-input').value = data.shortUrl;
    
    if (data.qrCode) {
        document.getElementById('qr-code-img').src = data.qrCode;
        document.getElementById('qr-code-display').classList.remove('hidden');
    }

    document.getElementById('url-stats-display').innerHTML = `
        <p><strong>Created:</strong> ${new Date().toLocaleString()}</p>
        <p><strong>Clicks:</strong> 0</p>
    `;

    document.getElementById('url-result').classList.remove('hidden');
    
    window.currentQRCode = data.qrCode;
}

function copyShortURL() {
    const input = document.getElementById('short-url-input');
    input.select();
    document.execCommand('copy');
    
    const btn = event.target;
    const originalText = btn.textContent;
    btn.textContent = '✅ Copied!';
    setTimeout(() => btn.textContent = originalText, 2000);
}

function downloadQR() {
    if (!window.currentQRCode) return;
    
    const link = document.createElement('a');
    link.href = window.currentQRCode;
    link.download = `qr_code_${Date.now()}.png`;
    link.click();
}

function addToHistory(data) {
    urlHistory.unshift({
        longUrl: data.originalUrl,
        shortUrl: data.shortUrl,
        alias: data.alias,
        timestamp: Date.now()
    });
    saveURLHistory();
    displayURLHistory();
}

function displayURLHistory() {
    const list = document.getElementById('url-history-list');
    if (urlHistory.length === 0) {
        list.innerHTML = '<p>No recent links</p>';
        return;
    }

    list.innerHTML = urlHistory.map(item => `
        <div class="history-item">
            <div>
                <strong>${item.shortUrl}</strong>
                <p>${item.longUrl}</p>
                <small>${new Date(item.timestamp).toLocaleString()}</small>
            </div>
            <button class="btn-small" onclick="copyToClipboard('${item.shortUrl}')">Copy</button>
        </div>
    `).join('');
}

function copyToClipboard(text) {
    navigator.clipboard.writeText(text);
    alert('Copied to clipboard!');
}


// Expose to window for router
window.initializeURLShortener = initializeURLShortener;
