// UNZIP Files Tool - Extract files from ZIP archives
let unzipFile = null;
let zipEntries = [];

// Load JSZip library
function loadJSZip() {
    return new Promise((resolve, reject) => {
        if (window.JSZip) {
            resolve();
            return;
        }

        const script = document.createElement('script');
        script.src = 'https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js';
        script.async = true;
        script.onload = () => resolve();
        script.onerror = () => reject(new Error('Failed to load JSZip library'));
        document.head.appendChild(script);
    });
}

window.initializeUnzipFiles = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-file-archive"></i> Extract ZIP Archive</h2>
                <p>Extract and download files from ZIP archives</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="unzip-upload-zone">
                    <i class="fas fa-file-archive"></i>
                    <h3>Click to select ZIP file</h3>
                    <p>Upload a ZIP archive to extract</p>
                    <input type="file" id="unzip-file-input" accept=".zip" style="display: none;">
                </div>
            </div>

            <div id="unzip-file-list" class="unzip-file-list hidden"></div>

            <div id="unzip-actions" class="action-buttons hidden">
                <button onclick="clearUnzipFile()" class="btn-secondary">
                    <i class="fas fa-trash-alt"></i> Clear
                </button>
                <button onclick="extractAllFiles()" class="btn-primary">
                    <i class="fas fa-download"></i> Extract All
                </button>
            </div>

            <div id="unzip-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('unzip-upload-zone').onclick = () => {
        document.getElementById('unzip-file-input').click();
    };

    document.getElementById('unzip-file-input').addEventListener('change', handleUnzipFileUpload);
    
    // Drag and drop
    const uploadZone = document.getElementById('unzip-upload-zone');
    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.classList.add('dragover');
    });
    
    uploadZone.addEventListener('dragleave', () => {
        uploadZone.classList.remove('dragover');
    });
    
    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.classList.remove('dragover');
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            handleUnzipFileUpload({ target: { files: files } });
        }
    });
};

async function handleUnzipFileUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    if (!file.name.endsWith('.zip')) {
        alert('Please select a ZIP file');
        return;
    }

    const listDiv = document.getElementById('unzip-file-list');
    listDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Reading ZIP archive...</h3></div>';
    listDiv.classList.remove('hidden');

    try {
        await loadJSZip();

        const arrayBuffer = await file.arrayBuffer();
        const zip = await JSZip.loadAsync(arrayBuffer);
        
        unzipFile = file;
        zipEntries = [];
        
        // Get all files in the ZIP
        zip.forEach((relativePath, zipEntry) => {
            if (!zipEntry.dir) { // Skip directories
                zipEntries.push({
                    name: relativePath,
                    entry: zipEntry,
                    size: zipEntry._data ? zipEntry._data.uncompressedSize : 0
                });
            }
        });

        displayUnzipFileList(zip);
    } catch (error) {
        alert('Failed to read ZIP file: ' + error.message);
        listDiv.classList.add('hidden');
    }
}

function displayUnzipFileList(zip) {
    const listDiv = document.getElementById('unzip-file-list');
    const actionsDiv = document.getElementById('unzip-actions');
    
    if (zipEntries.length === 0) {
        listDiv.innerHTML = '<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Empty ZIP Archive</h3><p>No files found in this archive</p></div>';
        actionsDiv.classList.add('hidden');
        return;
    }
    
    listDiv.classList.remove('hidden');
    actionsDiv.classList.remove('hidden');
    
    let totalSize = 0;
    let html = `<h3>📦 Files in Archive (${zipEntries.length})</h3>`;
    html += `<div class="file-thumbnail-grid">`;
    
    zipEntries.forEach((entry, index) => {
        const size = (entry.size / 1024).toFixed(1);
        totalSize += entry.size;
        const icon = getFileIcon(entry.name);
        
        html += `
            <div class="file-thumbnail-item">
                <button onclick="extractSingleFile(${index})" class="file-thumbnail-remove" title="Extract" style="background: rgba(34, 197, 94, 0.9);">
                    <i class="fas fa-download"></i>
                </button>
                <div class="file-thumbnail-preview">
                    <i class="fas ${icon} file-thumbnail-icon"></i>
                </div>
                <div class="file-thumbnail-name" title="${entry.name}">${entry.name}</div>
                <div class="file-thumbnail-info">${size} KB</div>
            </div>
        `;
    });

    const totalSizeMB = (totalSize / 1024 / 1024).toFixed(2);
    html += '</div>';
    html += `<div style="text-align: center; margin-top: 1rem; color: var(--text-secondary);">Total size: ${totalSizeMB} MB</div>`;
    listDiv.innerHTML = html;
}

function getFileIcon(filename) {
    const ext = filename.split('.').pop().toLowerCase();
    const iconMap = {
        'pdf': 'fa-file-pdf',
        'doc': 'fa-file-word', 'docx': 'fa-file-word',
        'xls': 'fa-file-excel', 'xlsx': 'fa-file-excel',
        'ppt': 'fa-file-powerpoint', 'pptx': 'fa-file-powerpoint',
        'jpg': 'fa-file-image', 'jpeg': 'fa-file-image', 'png': 'fa-file-image', 'gif': 'fa-file-image',
        'mp4': 'fa-file-video', 'avi': 'fa-file-video', 'mov': 'fa-file-video',
        'mp3': 'fa-file-audio', 'wav': 'fa-file-audio',
        'zip': 'fa-file-archive', 'rar': 'fa-file-archive', '7z': 'fa-file-archive',
        'txt': 'fa-file-alt', 'html': 'fa-file-code', 'css': 'fa-file-code', 'js': 'fa-file-code'
    };
    return iconMap[ext] || 'fa-file';
}

async function extractSingleFile(index) {
    const entry = zipEntries[index];
    
    try {
        const blob = await entry.entry.async('blob');
        const url = URL.createObjectURL(blob);
        
        const a = document.createElement('a');
        a.href = url;
        a.download = entry.name;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    } catch (error) {
        alert('Failed to extract file: ' + error.message);
    }
}

async function extractAllFiles() {
    const resultDiv = document.getElementById('unzip-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Extracting files...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        let extracted = 0;
        
        for (const entry of zipEntries) {
            const blob = await entry.entry.async('blob');
            const url = URL.createObjectURL(blob);
            
            const a = document.createElement('a');
            a.href = url;
            a.download = entry.name;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            
            extracted++;
            
            // Small delay to prevent browser blocking multiple downloads
            await new Promise(resolve => setTimeout(resolve, 100));
        }

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ Files Extracted Successfully!</h3>
                <p>${extracted} files extracted from archive</p>
                <div style="margin-top: 1.5rem; text-align: center;">
                    <button onclick="clearUnzipFile()" class="btn-primary" style="padding: 0.75rem 1.5rem;">
                        <i class="fas fa-redo"></i> Extract Another ZIP
                    </button>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Extraction error:', error);
        resultDiv.innerHTML = `
            <div class="error-message">
                <i class="fas fa-exclamation-circle"></i>
                <h3>Error Extracting Files</h3>
                <p>${error.message}</p>
            </div>
        `;
    }
}

function clearUnzipFile() {
    unzipFile = null;
    zipEntries = [];
    document.getElementById('unzip-file-list').classList.add('hidden');
    document.getElementById('unzip-actions').classList.add('hidden');
    document.getElementById('unzip-result').classList.add('hidden');
    document.getElementById('unzip-file-input').value = '';
}
