console.log('Typing Speed Master script executing...');

function initializeTypingSpeedMaster() {
    console.log('Initializing Typing Speed Master...');
    
    const toolContent = document.getElementById('tool-container');
    if (!toolContent) {
        console.error('Tool container element not found');
        return;
    }

    // Word Bank
    const wordList = [
        "the","of","and","a","to","in","is","you","that","it","he","was","for","on","are","as","with","his","they","at","be","this","have","from","or","one","had","by","word","but","not","what","all","were","we","when","your","can","said","there","use","an","each","which","she","do","how","their","if","will","up","other","about","out","many","then","them","these","so","some","her","would","make","like","him","into","time","has","look","two","more","write","go","see","number","no","way","could","people","my","than","first","water","been","call","who","oil","its","now","find","long","down","day","did","get","come","made","may","part","over","new","sound","take","only","little","work","know","place","year","live","me","back","give","most","very","after","things","our","just","name","good","sentence","man","think","say","great","where","help","through","much","before","line","right","too","mean","old","any","same","tell","boy","follow","came","want","show","also","around","form","three","small","set","put","end","does","another","well","large","must","big","even","such","because","turn","here","why","ask","went","men","read","need","land","different","home","us","move","try","kind","hand","picture","again","change","off","play","spell","air","away","animal","house","point","page","letter","mother","answer","found","study","still","learn","should","america","world"
    ];

    toolContent.innerHTML = `
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }

            :root {
                --bg-color: #000000;
                --main-color: #e2b714;
                --sub-color: #646669;
                --text-color: #d1d0c5;
                --error-color: #ca4754;
                --error-extra-color: #7e2a33;
                --caret-color: #e2b714;
                --font: 'Roboto Mono', monospace;
                --primary-color: #4f46e5;
                --gradient-primary: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                --text-secondary: #6b7280;
                --border-light: #e5e7eb;
            }

            .typing-master-wrapper {
                background-color: var(--bg-color);
                color: var(--text-color);
                font-family: var(--font);
                width: 100%;
                min-height: 100vh;
                display: flex;
                flex-direction: column;
                align-items: center;
                justify-content: center;
                overflow-x: hidden;
                transition: 0.2s;
                position: relative;
            }

            /* Time Selection Screen */
            .time-selection-wrapper {
                display: flex;
                flex-direction: column;
                justify-content: center;
                align-items: center;
                text-align: center;
                width: 100%;
                height: 100vh;
                padding: 20px;
                gap: 40px;
                background-color: #000000;
            }

            .time-selection-heading {
                color: var(--main-color);
                font-size: 32px;
                font-weight: 600;
                letter-spacing: 1px;
            }

            .time-buttons-grid {
                display: grid;
                grid-template-columns: repeat(4, 100px);
                gap: 20px;
                justify-content: center;
            }

            .time-btn {
                background: transparent;
                border: 2px solid var(--sub-color);
                color: var(--text-color);
                padding: 20px 15px;
                font-size: 18px;
                border-radius: 8px;
                cursor: pointer;
                transition: all 0.3s ease;
                font-family: var(--font);
                font-weight: 600;
            }

            .time-btn:hover {
                border-color: var(--main-color);
                color: var(--main-color);
                box-shadow: 0 0 10px rgba(226, 183, 20, 0.3);
                transform: translateY(-2px);
            }

            .time-btn:active {
                transform: scale(0.95);
            }

            /* Header */
            nav {
                display: none;
            }

            .logo {
                display: none;
            }

            .logo i {
                font-size: 2.2rem;
                background: var(--gradient-primary);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
            }

            .logo span { 
                font-size: 1.5rem;
                font-weight: 800;
                background: var(--gradient-primary);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
            }

            /* Main Typing Area */
            .game-container {
                position: relative;
                width: 100%;
                max-width: 1000px;
                height: 160px;
                overflow: hidden;
                font-size: 28px;
                line-height: 1.5;
            }

            .words-wrapper {
                display: flex;
                flex-wrap: wrap;
                user-select: none;
                transition: margin-top 0.25s ease;
            }

            .word {
                margin: 5px 15px 5px 0;
                position: relative;
                display: flex;
            }

            .letter {
                position: relative;
                color: var(--sub-color);
                transition: color 0.1s;
            }

            .letter.correct { color: var(--text-color); }
            .letter.incorrect { color: var(--error-color); }

            /* Blinking Cursor */
            #caret {
                position: absolute;
                width: 3px;
                height: 35px;
                background-color: var(--caret-color);
                border-radius: 2px;
                transition: left 0.1s ease, top 0.1s ease;
                animation: pulse 1s infinite;
                z-index: 9;
                pointer-events: none;
            }

            @keyframes pulse {
                0%, 100% { opacity: 1; }
                50% { opacity: 0; }
            }

            /* Hidden Input */
            #input-field {
                position: absolute;
                opacity: 0;
                top: 0;
                left: 0;
                cursor: default;
            }

            /* Live HUD */
            .hud {
                width: 1000px;
                display: flex;
                justify-content: space-between;
                color: var(--main-color);
                font-size: 20px;
                margin-bottom: 10px;
                opacity: 1;
                transition: opacity 0.3s;
            }
            
            /* Results Screen */
            #result-screen {
                display: none;
                flex-direction: column;
                align-items: center;
                width: 100%;
                max-width: 1000px;
                animation: fadeIn 0.3s forwards;
            }

            .stats-grid {
                display: grid;
                grid-template-columns: repeat(4, 1fr);
                gap: 20px;
                width: 100%;
                margin-bottom: 30px;
            }

            .stat-item {
                background: transparent;
                display: flex;
                flex-direction: column;
            }
            
            .stat-label { color: var(--sub-color); font-size: 16px; margin-bottom: 5px; }
            .stat-value { color: var(--main-color); font-size: 42px; font-weight: bold; }
            .stat-value.secondary { font-size: 24px; color: var(--text-color); }

            .chart-container {
                width: 100%;
                height: 300px;
                background: #2c2e31;
                border-radius: 10px;
                padding: 15px;
                margin-bottom: 20px;
            }

            /* Restart Button */
            .restart-btn {
                background: var(--main-color);
                color: var(--bg-color);
                border: none;
                padding: 12px 35px;
                font-family: var(--font);
                font-size: 16px;
                border-radius: 6px;
                cursor: pointer;
                margin-top: 20px;
                transition: all 0.2s ease;
                font-weight: bold;
            }
            .restart-btn:hover { 
                background: #f5c946;
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(226, 183, 20, 0.3);
            }
            .restart-btn:active {
                transform: translateY(0);
            }

            /* Footer Tooltip */
            footer {
                margin-top: 50px;
                color: var(--sub-color);
                font-size: 12px;
                display: flex;
                justify-content: center;
                align-items: center;
            }

            /* Restart Icon Button */
            .restart-icon-btn {
                background: none;
                border: none;
                outline: none;
                color: var(--main-color);
                font-size: 24px;
                cursor: pointer;
                padding: 8px;
                border-radius: 6px;
                transition: all 0.3s ease;
                display: flex;
                align-items: center;
                justify-content: center;
            }

            .restart-icon-btn:hover {
                color: #f5c946;
                transform: rotate(180deg);
            }

            .restart-icon-btn:active {
                transform: rotate(180deg) scale(0.95);
            }

            .key {
                background-color: var(--sub-color);
                color: var(--bg-color);
                padding: 2px 6px;
                border-radius: 3px;
                font-weight: bold;
            }

            @keyframes fadeIn { from { opacity: 0; transform: translateY(20px); } to { opacity: 1; transform: translateY(0); } }

            @media (max-width: 1024px) {
                .time-buttons-grid {
                    grid-template-columns: repeat(3, 90px);
                }

                .stats-grid {
                    grid-template-columns: repeat(2, 1fr);
                }

                .hud { width: 900px; }
                .game-container { max-width: 900px; }
            }

            @media (max-width: 768px) {
                .time-buttons-grid {
                    grid-template-columns: repeat(2, 80px);
                }

                .stats-grid {
                    grid-template-columns: repeat(2, 1fr);
                }

                nav { max-width: 900px; }
                .hud { width: 900px; }
                .game-container { max-width: 900px; }
                #result-screen { max-width: 900px; }
            }

            @media (max-width: 480px) {
                .time-buttons-grid {
                    grid-template-columns: 1fr;
                    max-width: 150px;
                }

                .stats-grid {
                    grid-template-columns: 1fr;
                }

                nav { max-width: 500px; }
                .hud { width: 500px; }
                .game-container { max-width: 500px; }
                #result-screen { max-width: 500px; }
            }
        </style>

        <!-- Time Selection Screen (Duration Page) -->
        <div class="time-selection-wrapper" id="time-selection-screen">
            <h1 class="time-selection-heading">Select Duration</h1>
            <div class="time-buttons-grid">
                <button class="time-btn" data-time="15">15s</button>
                <button class="time-btn" data-time="30">30s</button>
                <button class="time-btn" data-time="60">1m</button>
                <button class="time-btn" data-time="120">2m</button>
                <button class="time-btn" data-time="180">3m</button>
                <button class="time-btn" data-time="300">5m</button>
                <button class="time-btn" data-time="600">10m</button>
                <button class="time-btn" data-time="900">15m</button>
                <button class="time-btn" data-time="2700">45m</button>
                <button class="time-btn" data-time="1800">30m</button>
                <button class="time-btn" data-time="3600">1h</button>
                <button class="time-btn" data-time="7200">2h</button>
            </div>
        </div>

        <!-- Game Content (Hidden initially) -->
        <div class="typing-master-wrapper" id="game-wrapper" style="display: none;">
            <!-- Header -->
            <nav id="nav">
                <a href="/" class="logo" style="text-decoration: none;">
                    <i class="fas fa-file-pdf"></i>
                    <span>PDF & Image Toolkit</span>
                </a>
            </nav>

            <!-- Live Timer/Count -->
            <div class="hud" id="hud">
                <div id="timer">30</div>
                <div id="live-wpm">0 WPM</div>
            </div>

            <!-- Typing Game -->
            <div class="game-container" id="game-ui" onclick="window.focusInput()">
                <div id="caret"></div>
                <div class="words-wrapper" id="words"></div>
            </div>
            
            <input type="text" id="input-field" autocomplete="off" spellcheck="false">

            <!-- Results Screen -->
            <div id="result-screen">
                <div class="stats-grid">
                    <div class="stat-item">
                        <span class="stat-label">wpm</span>
                        <span class="stat-value" id="res-wpm">0</span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">acc</span>
                        <span class="stat-value" id="res-acc">0%</span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">raw</span>
                        <span class="stat-value secondary" id="res-raw">0</span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">characters</span>
                        <span class="stat-value secondary" id="res-chars">0/0/0</span>
                    </div>
                </div>
                <div class="chart-container">
                    <canvas id="wpmChart"></canvas>
                </div>
                <button class="restart-btn" onclick="window.typingMasterRestart()"><i class="fas fa-redo"></i> Restart Test</button>
            </div>

            <footer>
                <button class="restart-icon-btn" onclick="window.typingMasterRestart()" title="Restart Test"><i class="fas fa-redo"></i></button>
            </footer>
        </div>
    `;

    // Add Google Fonts
    if (!document.querySelector('link[href*="Roboto+Mono"]')) {
        const link = document.createElement('link');
        link.href = "https://fonts.googleapis.com/css2?family=Roboto+Mono:wght@400;500;700&display=swap";
        link.rel = "stylesheet";
        document.head.appendChild(link);
    }

    // Add Font Awesome
    if (!document.querySelector('link[href*="font-awesome"]')) {
        const faLink = document.createElement('link');
        faLink.href = "https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css";
        faLink.rel = "stylesheet";
        document.head.appendChild(faLink);
    }

    // Add Chart.js
    if (!window.Chart) {
        const script = document.createElement('script');
        script.src = "https://cdn.jsdelivr.net/npm/chart.js";
        document.head.appendChild(script);
    }

    // --- DOM ELEMENTS ---
    const wordsContainer = toolContent.querySelector('#words');
    const inputField = toolContent.querySelector('#input-field');
    const caret = toolContent.querySelector('#caret');
    const timerEl = toolContent.querySelector('#timer');
    const liveWpmEl = toolContent.querySelector('#live-wpm');
    const hud = toolContent.querySelector('#hud');
    const gameUi = toolContent.querySelector('#game-ui');
    const resultScreen = toolContent.querySelector('#result-screen');
    const nav = toolContent.querySelector('#nav');
    const timeSelectionScreen = toolContent.querySelector('#time-selection-screen');
    const timeButtons = toolContent.querySelectorAll('.time-btn');

    // --- GAME STATE ---
    let timer = null;
    let maxTime = 30;
    let timeLeft = maxTime;
    let gameActive = false;
    let isTyping = false;
    let currentWordIndex = 0;
    let currentLetterIndex = 0;
    let correctChars = 0;
    let incorrectChars = 0;
    let extraChars = 0;
    let historyWPM = [];
    let wordsArray = [];

    // Time selection
    const gameWrapper = toolContent.querySelector('#game-wrapper');
    timeButtons.forEach(btn => {
        btn.addEventListener('click', () => {
            maxTime = parseInt(btn.getAttribute('data-time'));
            timeLeft = maxTime;
            timerEl.innerText = maxTime;
            timeSelectionScreen.style.display = 'none';
            gameWrapper.style.display = 'flex';
            initGame();
        });
    });

    // --- INITIALIZATION ---
    function initGame() {
        // Reset UI
        wordsContainer.innerHTML = "";
        wordsContainer.style.marginTop = "0px";
        inputField.value = "";
        timerEl.innerText = maxTime;
        liveWpmEl.innerText = "";
        resultScreen.style.display = "none";
        gameUi.style.display = "block";
        hud.style.opacity = "1";
        nav.style.opacity = "1";
        
        // Reset State
        timeLeft = maxTime;
        gameActive = true;
        isTyping = false;
        currentWordIndex = 0;
        currentLetterIndex = 0;
        correctChars = 0;
        incorrectChars = 0;
        extraChars = 0;
        historyWPM = [];
        clearInterval(timer);

        // Generate Words (Start with more for longer tests)
        wordsArray = [];
        const wordCount = Math.max(200, Math.ceil((maxTime / 60) * 50)); // Dynamic word count
        for (let i = 0; i < wordCount; i++) {
            const rand = wordList[Math.floor(Math.random() * wordList.length)];
            wordsArray.push(rand);
        }
        
        // Render Words
        renderWords();
        updateCaretPosition();
        window.focusInput();
    }
    
    // Function to render words to the DOM
    function renderWords() {
        wordsArray.forEach(word => {
            const wordDiv = document.createElement('div');
            wordDiv.className = 'word';
            word.split('').forEach(char => {
                const span = document.createElement('span');
                span.className = 'letter';
                span.innerText = char;
                wordDiv.appendChild(span);
            });
            wordsContainer.appendChild(wordDiv);
        });
    }
    
    // Function to add more words dynamically (endless text)
    function addMoreWords() {
        const wordsToAdd = 50;
        for (let i = 0; i < wordsToAdd; i++) {
            const rand = wordList[Math.floor(Math.random() * wordList.length)];
            wordsArray.push(rand);
        }
        
        // Render only the new words
        const startIndex = wordsArray.length - wordsToAdd;
        for (let i = startIndex; i < wordsArray.length; i++) {
            const word = wordsArray[i];
            const wordDiv = document.createElement('div');
            wordDiv.className = 'word';
            word.split('').forEach(char => {
                const span = document.createElement('span');
                span.className = 'letter';
                span.innerText = char;
                wordDiv.appendChild(span);
            });
            wordsContainer.appendChild(wordDiv);
        }
    }

    // --- CORE TYPING LOGIC ---
    inputField.addEventListener('keydown', (e) => {
        if (!gameActive) return;

        // Start Timer on first valid key
        if (!isTyping && "abcdefghijklmnopqrstuvwxyz ".includes(e.key.toLowerCase())) {
            isTyping = true;
            startTimer();
            hud.style.opacity = "0.5";
            nav.style.opacity = "0.5";
        }
    });

    inputField.addEventListener('input', (e) => {
        if (!gameActive) return;

        const value = inputField.value;
        const words = wordsContainer.querySelectorAll('.word');
        const currentWordDiv = words[currentWordIndex];
        const currentLetters = currentWordDiv.querySelectorAll('.letter');

        const inputChar = value[value.length - 1];
        const inputType = e.inputType;

        if (inputType === "deleteContentBackward") {
            if (currentLetterIndex > 0) {
                currentLetterIndex--;
                if (currentLetterIndex >= currentLetters.length) {
                    const extras = currentWordDiv.querySelectorAll('.letter.extra');
                    if (extras.length > 0) {
                        extras[extras.length - 1].remove();
                    }
                } else {
                    currentLetters[currentLetterIndex].className = 'letter';
                }
            }
        } else if (inputChar === " ") {
            for (let i = currentLetterIndex; i < currentLetters.length; i++) {
                currentLetters[i].classList.add('incorrect');
                incorrectChars++;
            }

            currentWordIndex++;
            currentLetterIndex = 0;
            inputField.value = "";
            
            // Generate more words if running low (endless text)
            const words = wordsContainer.querySelectorAll('.word');
            if (currentWordIndex > wordsArray.length - 100) {
                addMoreWords();
            }
            
            // Smart scroll: only scroll if next word is not visible
            const nextWord = words[currentWordIndex];
            if (nextWord) {
                const gameUIRect = gameUi.getBoundingClientRect();
                const nextWordRect = nextWord.getBoundingClientRect();
                const relativeTop = nextWordRect.top - gameUIRect.top;
                
                // If word is below visible area, scroll up
                if (relativeTop + nextWordRect.height > gameUi.offsetHeight) {
                    const lineHeight = Math.ceil(nextWordRect.height) + 10; // word height + margin
                    const currentMargin = parseInt(wordsContainer.style.marginTop || 0);
                    wordsContainer.style.marginTop = (currentMargin - lineHeight) + "px";
                }
            }
            
            // Update caret position AFTER scroll
            updateCaretPosition();
            return;
        } else {
            if (currentLetterIndex < currentLetters.length) {
                if (inputChar === currentLetters[currentLetterIndex].innerText) {
                    currentLetters[currentLetterIndex].classList.add('correct');
                    correctChars++;
                } else {
                    currentLetters[currentLetterIndex].classList.add('incorrect');
                    incorrectChars++;
                }
                currentLetterIndex++;
            } else {
                const span = document.createElement('span');
                span.className = 'letter incorrect extra';
                span.innerText = inputChar;
                currentWordDiv.appendChild(span);
                currentLetterIndex++;
                extraChars++;
                incorrectChars++;
            }
        }

        updateCaretPosition();
    });

    // --- CARET MOVEMENT ---
    function updateCaretPosition() {
        const words = wordsContainer.querySelectorAll('.word');
        const currentWordDiv = words[currentWordIndex];
        const currentLetters = currentWordDiv.querySelectorAll('.letter');
        
        let rect;

        if (currentLetterIndex === 0) {
            rect = currentWordDiv.getBoundingClientRect();
            moveCaret(rect.left - 2, rect.top + 5);
        } else if (currentLetterIndex > 0) {
            const prevCharIndex = currentLetterIndex - 1;
            const prevChar = currentLetters[prevCharIndex];
            
            if(prevChar) {
                rect = prevChar.getBoundingClientRect();
                moveCaret(rect.right, rect.top);
            }
        }
    }

    function moveCaret(x, y) {
        const containerRect = gameUi.getBoundingClientRect();
        caret.style.left = (x - containerRect.left) + "px";
        caret.style.top = (y - containerRect.top + 4) + "px";
        
        caret.style.animation = 'none';
        caret.offsetHeight;
        caret.style.animation = 'pulse 1s infinite';
    }

    // --- TIMER & GAME OVER ---
    function startTimer() {
        timer = setInterval(() => {
            timeLeft--;
            timerEl.innerText = timeLeft;
            
            // Calculate Live WPM with professional formula
            const timeElapsedSec = maxTime - timeLeft;
            const timeElapsedMin = timeElapsedSec / 60;
            const wpm = Math.round((correctChars / 5) / timeElapsedMin);
            liveWpmEl.innerText = (wpm || 0) + " WPM";
            
            historyWPM.push(wpm || 0);

            if (timeLeft <= 0) {
                gameOver();
            }
        }, 1000);
    }

    function gameOver() {
        clearInterval(timer);
        gameActive = false;
        
        gameUi.style.display = "none";
        hud.style.opacity = "0";
        nav.style.opacity = "1";
        resultScreen.style.display = "flex";

        // Calculate Stats with professional formulas
        const timeElapsedMin = maxTime / 60;
        const netWPM = Math.round((correctChars / 5) / timeElapsedMin);
        const grossWPM = Math.round(((correctChars + incorrectChars) / 5) / timeElapsedMin);
        const totalTyped = correctChars + incorrectChars + extraChars;
        const accuracy = totalTyped > 0 ? Math.round((correctChars / totalTyped) * 100) : 0;

        document.getElementById('res-wpm').innerText = netWPM;
        document.getElementById('res-acc').innerText = accuracy + "%";
        document.getElementById('res-raw').innerText = grossWPM;
        document.getElementById('res-chars').innerText = `${correctChars}/${incorrectChars}/${extraChars}`;

        renderChart();
    }

    // --- CHART.JS INTEGRATION ---
    let wpmChartInstance = null;
    function renderChart() {
        const canvasEl = toolContent.querySelector('#wpmChart');
        if (!canvasEl) return;

        const ctx = canvasEl.getContext('2d');
        const labels = Array.from({length: historyWPM.length}, (_, i) => i + 1);

        if(wpmChartInstance) wpmChartInstance.destroy();

        setTimeout(() => {
            if (window.Chart) {
                wpmChartInstance = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: labels,
                        datasets: [{
                            label: 'WPM',
                            data: historyWPM,
                            borderColor: '#e2b714',
                            backgroundColor: 'rgba(226, 183, 20, 0.1)',
                            borderWidth: 2,
                            tension: 0.4,
                            fill: true,
                            pointRadius: 2,
                            pointBackgroundColor: '#e2b714'
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: {
                            y: { beginAtZero: true, grid: { color: '#444' }, ticks: { color: '#d1d0c5' } },
                            x: { grid: { display: false }, ticks: { color: '#d1d0c5' } }
                        }
                    }
                });
            }
        }, 100);
    }

    // --- UTILS ---
    window.focusInput = function() {
        inputField.focus();
    };

    window.typingMasterRestart = function() {
        gameWrapper.style.display = 'none';
        timeSelectionScreen.style.display = 'flex';
        resultScreen.style.display = "none";
        gameUi.style.display = "block";
        inputField.value = "";
    };

    window.addEventListener('resize', () => {
        updateCaretPosition();
    });

    // Export initialization
    window.initializeTypingSpeedMaster = initializeTypingSpeedMaster;
}

// Attach to window
window.initializeTypingSpeedMaster = initializeTypingSpeedMaster;
console.log('Typing Speed Master initialized');
