// Text Extractor from Images (OCR) Tool
function initializeTextExtractor() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="text-extractor-container">
            <h2>📝 Text Extractor (OCR)</h2>
            <p class="tool-description">Extract text from images and PDFs using OCR</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="ocr-input" accept="image/*,.pdf" onchange="loadFileForOCR(this)">
                    <span>📁 Choose Image or PDF</span>
                </label>
            </div>

            <div id="ocr-preview" class="preview-section hidden">
                <h3>Source File</h3>
                <img id="ocr-preview-img" src="" alt="Preview" style="max-width: 400px;">
            </div>

            <div id="ocr-options" class="ocr-options hidden">
                <div class="form-group">
                    <label>Language:</label>
                    <select id="ocr-language">
                        <option value="eng">English</option>
                        <option value="spa">Spanish</option>
                        <option value="fra">French</option>
                        <option value="deu">German</option>
                        <option value="ita">Italian</option>
                        <option value="por">Portuguese</option>
                        <option value="rus">Russian</option>
                        <option value="ara">Arabic</option>
                        <option value="chi_sim">Chinese (Simplified)</option>
                        <option value="jpn">Japanese</option>
                    </select>
                </div>

                <button class="btn btn-primary" onclick="extractText()">Extract Text</button>
            </div>

            <div id="ocr-result" class="result-section hidden">
                <h3>Extracted Text</h3>
                <textarea id="extracted-text" rows="15" readonly></textarea>
                
                <div class="button-group">
                    <button class="btn btn-secondary" onclick="copyExtractedText()">📋 Copy Text</button>
                    <button class="btn btn-secondary" onclick="downloadText()">💾 Download as TXT</button>
                    <button class="btn btn-secondary" onclick="downloadDocx()">📄 Download as DOCX</button>
                </div>

                <div class="text-stats">
                    <p><strong>Words:</strong> <span id="word-count">0</span></p>
                    <p><strong>Characters:</strong> <span id="char-count">0</span></p>
                </div>
            </div>
        </div>
    `;
}

let ocrFile = null;

function loadFileForOCR(input) {
    ocrFile = input.files[0];
    if (!ocrFile) return;

    if (ocrFile.type.startsWith('image/')) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('ocr-preview-img').src = e.target.result;
            document.getElementById('ocr-preview').classList.remove('hidden');
        };
        reader.readAsDataURL(ocrFile);
    }

    document.getElementById('ocr-options').classList.remove('hidden');
}

async function extractText() {
    if (!ocrFile) return;

    // Show placeholder text instead of making API call
    const placeholderText = `This is placeholder text extracted from your file.\n\nOCR (Optical Character Recognition) requires additional dependencies like Tesseract.js.\n\nIn a production environment, this would:\n- Extract text from images using OCR\n- Extract text from PDFs\n- Support multiple languages\n- Provide accurate text recognition\n- Preserve formatting where possible\n\nFile: ${ocrFile.name}\nLanguage: ${document.getElementById('ocr-language').options[document.getElementById('ocr-language').selectedIndex].text}\n\nFor now, this is a demonstration of the interface.`;
    
    setTimeout(() => {
        displayExtractedText(placeholderText);
    }, 800);
}

function displayExtractedText(text) {
    document.getElementById('extracted-text').value = text;
    
    const words = text.trim().split(/\s+/).length;
    const chars = text.length;
    
    document.getElementById('word-count').textContent = words;
    document.getElementById('char-count').textContent = chars;
    
    document.getElementById('ocr-result').classList.remove('hidden');
}

function copyExtractedText() {
    const textarea = document.getElementById('extracted-text');
    textarea.select();
    document.execCommand('copy');
    
    const btn = event.target;
    const originalText = btn.textContent;
    btn.textContent = '✅ Copied!';
    setTimeout(() => btn.textContent = originalText, 2000);
}

function downloadText() {
    const text = document.getElementById('extracted-text').value;
    const blob = new Blob([text], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    
    const link = document.createElement('a');
    link.href = url;
    link.download = `extracted_text_${Date.now()}.txt`;
    link.click();
    
    URL.revokeObjectURL(url);
}

function downloadDocx() {
    alert('DOCX export functionality coming soon!');
}


// Expose to window for router
window.initializeTextExtractor = initializeTextExtractor;
