// SVG to PNG Converter

let svgFiles = [];

window.initializeSvgToPng = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-vector-square"></i> SVG to PNG Converter</h2>
                <p>Convert vector SVG to raster PNG images</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('svg-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop SVG files here or click to upload</h3>
                <p>Supports: SVG</p>
                <input type="file" id="svg-input" accept=".svg" multiple style="display: none;">
            </div>

            <div id="svg-preview" class="preview-section hidden">
                <h3>Selected Files (<span id="svg-count">0</span>)</h3>
                <div id="svg-list" class="file-list"></div>
            </div>

            <div id="svg-options" class="options-section hidden">
                <h3>Conversion Options</h3>
                <div class="form-group">
                    <label>Width (px):</label>
                    <input type="number" id="svg-width" min="1" value="800" class="form-input">
                </div>
                <div class="form-group">
                    <label>Height (px):</label>
                    <input type="number" id="svg-height" min="1" value="600" class="form-input">
                    <label>
                        <input type="checkbox" id="svg-maintain-ratio" checked> Maintain aspect ratio
                    </label>
                </div>
                <div class="form-group">
                    <label>Background:</label>
                    <select id="svg-background">
                        <option value="transparent">Transparent</option>
                        <option value="white">White</option>
                        <option value="black">Black</option>
                        <option value="custom">Custom Color</option>
                    </select>
                    <input type="color" id="svg-bg-color" value="#ffffff" class="hidden">
                </div>
                <button onclick="convertSvgToPng()" class="btn-primary">
                    <i class="fas fa-sync"></i> Convert to PNG
                </button>
            </div>

            <div id="svg-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('svg-input').addEventListener('change', handleSvgUpload);
    document.getElementById('svg-background').addEventListener('change', toggleSvgBgColor);
};

function handleSvgUpload(e) {
    svgFiles = Array.from(e.target.files);
    if (!svgFiles.length) return;

    document.getElementById('svg-count').textContent = svgFiles.length;
    document.getElementById('svg-preview').classList.remove('hidden');
    document.getElementById('svg-options').classList.remove('hidden');

    const list = document.getElementById('svg-list');
    list.innerHTML = '';

    svgFiles.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'file-item';
        item.innerHTML = `
            <i class="fas fa-file-code"></i>
            <span>${file.name}</span>
            <span class="file-size">${(file.size / 1024).toFixed(2)} KB</span>
        `;
        list.appendChild(item);
    });
}

function toggleSvgBgColor() {
    const bg = document.getElementById('svg-background').value;
    document.getElementById('svg-bg-color').classList.toggle('hidden', bg !== 'custom');
}

async function convertSvgToPng() {
    if (!svgFiles.length) return;

    const resultsDiv = document.getElementById('svg-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Converting SVG to PNG...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    svgFiles.forEach(file => formData.append('files', file));
    formData.append('width', document.getElementById('svg-width').value);
    formData.append('height', document.getElementById('svg-height').value);
    formData.append('maintainRatio', document.getElementById('svg-maintain-ratio').checked);
    
    const bgType = document.getElementById('svg-background').value;
    if (bgType === 'custom') {
        formData.append('backgroundColor', document.getElementById('svg-bg-color').value);
    } else {
        formData.append('backgroundColor', bgType);
    }

    try {
        const response = await fetch('/api/convert/svg-to-png', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displaySvgResults(data.images);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displaySvgResults(images) {
    const resultsDiv = document.getElementById('svg-results');
    let html = '<div class="results-success"><h3>✅ Conversion Complete!</h3><div class="results-grid">';

    images.forEach(img => {
        html += `
            <div class="result-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="result-info">
                    <div class="result-name">${img.filename}</div>
                    <a href="${img.url}" download="${img.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download
                    </a>
                </div>
            </div>
        `;
    });

    html += '</div></div>';
    resultsDiv.innerHTML = html;
}
