// Split PDF Tool - Client-Side with PDF-lib and PDF.js
let splitPdfFile = null;
let splitPdfDoc = null;
let splitMode = 'pages';
let selectedPages = new Set();

window.initializeSplitPdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-cut"></i> Split PDF</h2>
                <p>Extract specific pages or split PDF into multiple files</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="split-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <p>Upload a PDF to split</p>
                    <input type="file" id="split-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="split-options" class="split-options hidden">
                <h3>Select Split Mode</h3>
                <div class="split-mode-selector">
                    <button class="split-mode-btn active" onclick="setSplitMode('pages')">
                        <i class="fas fa-hand-pointer"></i>
                        <div>Select Pages</div>
                        <small>Choose specific pages</small>
                    </button>
                    <button class="split-mode-btn" onclick="setSplitMode('range')">
                        <i class="fas fa-arrows-alt-h"></i>
                        <div>Page Range</div>
                        <small>Extract page range</small>
                    </button>
                    <button class="split-mode-btn" onclick="setSplitMode('interval')">
                        <i class="fas fa-layer-group"></i>
                        <div>Every N Pages</div>
                        <small>Split at intervals</small>
                    </button>
                </div>

                <div id="range-input" class="hidden" style="margin-top: 1rem;">
                    <label>Page Range (e.g., 1-5, 10-15):</label>
                    <input type="text" id="page-range" class="form-input" placeholder="1-5">
                </div>

                <div id="interval-input" class="hidden" style="margin-top: 1rem;">
                    <label>Split every:</label>
                    <input type="number" id="page-interval" class="form-input" placeholder="5" min="1" value="1">
                    <small>pages</small>
                </div>
            </div>

            <div id="split-preview" class="hidden"></div>

            <div id="split-actions" class="action-buttons hidden">
                <button onclick="splitPdfNow()" class="btn-primary">
                    <i class="fas fa-cut"></i> Split PDF
                </button>
            </div>

            <div id="split-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('split-upload-zone').onclick = () => {
        document.getElementById('split-pdf-input').click();
    };

    document.getElementById('split-pdf-input').addEventListener('change', handleSplitPdfUpload);
};

async function handleSplitPdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        splitPdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        splitPdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        document.getElementById('split-options').classList.remove('hidden');
        await renderSplitPreview();
        document.getElementById('split-actions').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

async function renderSplitPreview() {
    const previewDiv = document.getElementById('split-preview');
    previewDiv.classList.remove('hidden');
    
    const pageCount = splitPdfDoc.getPageCount();
    
    previewDiv.innerHTML = `
        <h3>📄 PDF Preview (${pageCount} pages)</h3>
        <div class="pdf-pages-grid" id="split-pages-grid"></div>
    `;

    const loadingTask = pdfjsLib.getDocument(await splitPdfFile.arrayBuffer());
    const pdf = await loadingTask.promise;
    const pagesGrid = document.getElementById('split-pages-grid');

    for (let i = 1; i <= pageCount; i++) {
        const page = await pdf.getPage(i);
        const viewport = page.getViewport({ scale: 0.5 });
        
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        canvas.width = viewport.width;
        canvas.height = viewport.height;

        await page.render({ canvasContext: ctx, viewport }).promise;

        const pageWrapper = document.createElement('div');
        pageWrapper.className = 'pdf-page-wrapper';
        pageWrapper.dataset.pageNum = i;
        
        if (splitMode === 'pages') {
            pageWrapper.onclick = () => togglePageSelection(i, pageWrapper);
        }

        const pageNumber = document.createElement('div');
        pageNumber.className = 'page-number-badge';
        pageNumber.textContent = `Page ${i}`;

        pageWrapper.appendChild(canvas);
        pageWrapper.appendChild(pageNumber);
        pagesGrid.appendChild(pageWrapper);
    }
}

function setSplitMode(mode) {
    splitMode = mode;
    selectedPages.clear();
    
    document.querySelectorAll('.split-mode-btn').forEach(btn => btn.classList.remove('active'));
    event.target.closest('.split-mode-btn').classList.add('active');
    
    document.getElementById('range-input').classList.add('hidden');
    document.getElementById('interval-input').classList.add('hidden');
    
    if (mode === 'range') {
        document.getElementById('range-input').classList.remove('hidden');
    } else if (mode === 'interval') {
        document.getElementById('interval-input').classList.remove('hidden');
    }

    // Remove selection styling
    document.querySelectorAll('.pdf-page-wrapper').forEach(wrapper => {
        wrapper.classList.remove('selected');
    });
}

function togglePageSelection(pageNum, wrapper) {
    if (splitMode !== 'pages') return;

    if (selectedPages.has(pageNum)) {
        selectedPages.delete(pageNum);
        wrapper.classList.remove('selected');
    } else {
        selectedPages.add(pageNum);
        wrapper.classList.add('selected');
    }
}

async function splitPdfNow() {
    if (!splitPdfDoc) {
        alert('Please upload a PDF first');
        return;
    }

    try {
        let pagesToExtract = [];

        if (splitMode === 'pages') {
            if (selectedPages.size === 0) {
                alert('Please select at least one page');
                resultDiv.classList.add('hidden');
                return;
            }
            pagesToExtract = [Array.from(selectedPages).sort((a, b) => a - b)];
        } else if (splitMode === 'range') {
            const range = document.getElementById('page-range').value;
            const match = range.match(/(\d+)-(\d+)/);
            if (!match) {
                alert('Invalid range format. Use format like 1-5');
                resultDiv.classList.add('hidden');
                return;
            }
            const start = parseInt(match[1]);
            const end = parseInt(match[2]);
            const pages = [];
            for (let i = start; i <= end && i <= splitPdfDoc.getPageCount(); i++) {
                pages.push(i);
            }
            pagesToExtract = [pages];
        } else if (splitMode === 'interval') {
            const interval = parseInt(document.getElementById('page-interval').value) || 1;
            const totalPages = splitPdfDoc.getPageCount();
            for (let i = 1; i <= totalPages; i += interval) {
                const pages = [];
                for (let j = i; j < i + interval && j <= totalPages; j++) {
                    pages.push(j);
                }
                pagesToExtract.push(pages);
            }
        }

        // Create split PDFs
        const files = [];
        for (let i = 0; i < pagesToExtract.length; i++) {
            const newPdf = await PDFLib.PDFDocument.create();
            const pages = pagesToExtract[i];
            
            for (const pageNum of pages) {
                const [copiedPage] = await newPdf.copyPages(splitPdfDoc, [pageNum - 1]);
                newPdf.addPage(copiedPage);
            }

            const pdfBytes = await newPdf.save();
            const blob = new Blob([pdfBytes], { type: 'application/pdf' });
            const url = URL.createObjectURL(blob);
            files.push({
                url,
                name: `split_${i + 1}.pdf`,
                pages: pages.join(', '),
                size: (blob.size / 1024).toFixed(1)
            });
        }

        // Store results and navigate to result page
        sessionStorage.setItem('splitResult', JSON.stringify({
            fileCount: files.length,
            originalName: splitPdfFile.name,
            timestamp: Date.now()
        }));
        
        // Store file data
        sessionStorage.setItem('splitFiles', JSON.stringify(files));
        
        // Navigate to result page
        showSplitResultPage();
    } catch (error) {
        console.error('Split error:', error);
        alert('Error splitting PDF: ' + error.message);
    }
}

function showSplitResultPage() {
    const container = document.getElementById('tool-container');
    if (!container) {
        console.error('Tool container not found');
        alert('Error loading result page');
        return;
    }
    
    const resultData = JSON.parse(sessionStorage.getItem('splitResult'));
    if (!resultData) {
        console.error('No result data found');
        return;
    }
    
    const files = JSON.parse(sessionStorage.getItem('splitFiles'));
    if (!files || files.length === 0) {
        console.error('No split files found');
        return;
    }
    
    let fileListHtml = '';
    files.forEach(file => {
        fileListHtml += `
            <div class="split-file-card">
                <div class="file-icon">
                    <i class="fas fa-file-pdf"></i>
                </div>
                <div class="file-details">
                    <strong>${file.name}</strong>
                    <p>Pages: ${file.pages} • ${file.size} KB</p>
                </div>
                <button onclick="downloadSplitFile('${file.url}', '${file.name}')" class="btn-download-small">
                    <i class="fas fa-download"></i> Download
                </button>
            </div>
        `;
    });
    
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="result-page">
                <div class="result-success-card">
                    <button onclick="window.initializeSplitPdf()" class="result-back-btn" title="Go Back">
                        <i class="fas fa-arrow-left"></i>
                    </button>
                    <div class="result-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <h2>✨ PDF Split Successfully!</h2>
                    <div class="result-info">
                        <p><strong>${resultData.fileCount}</strong> files created</p>
                    </div>
                    <div class="split-files-list">
                        ${fileListHtml}
                    </div>
                </div>
            </div>
        </div>
    `;
}

function downloadSplitFile(url, filename) {
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    link.click();
}
