// Signature Generator

let signatureCanvas;
let signatureCtx;
let isDrawing = false;

window.initializeSignatureGenerator = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-signature"></i> Signature Generator</h2>
                <p>Create digital signatures</p>
            </div>

            <div class="signature-container">
                <div class="signature-tabs">
                    <button class="sig-tab-btn active" onclick="switchSigTab('draw')">Draw</button>
                    <button class="sig-tab-btn" onclick="switchSigTab('type')">Type</button>
                </div>

                <div id="draw-sig-tab" class="tab-content">
                    <canvas id="signature-canvas" width="600" height="200"></canvas>
                    <div class="button-group">
                        <button onclick="clearSignature()" class="btn-secondary">
                            <i class="fas fa-eraser"></i> Clear
                        </button>
                        <button onclick="downloadSignature()" class="btn-primary">
                            <i class="fas fa-download"></i> Download PNG
                        </button>
                    </div>
                </div>

                <div id="type-sig-tab" class="tab-content hidden">
                    <div class="form-group">
                        <label>Your Name:</label>
                        <input type="text" id="sig-name" class="form-input" placeholder="John Doe" oninput="updateTypedSignature()">
                    </div>
                    <div class="form-group">
                        <label>Font Style:</label>
                        <select id="sig-font" onchange="updateTypedSignature()">
                            <option value="'Brush Script MT', cursive">Brush Script</option>
                            <option value="'Lucida Handwriting', cursive">Lucida Handwriting</option>
                            <option value="'Pacifico', cursive">Pacifico</option>
                            <option value="'Dancing Script', cursive">Dancing Script</option>
                            <option value="'Great Vibes', cursive">Great Vibes</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Size: <span id="sig-size-value">60</span>px</label>
                        <input type="range" id="sig-size" min="30" max="120" value="60" oninput="updateTypedSignature(); document.getElementById('sig-size-value').textContent = this.value">
                    </div>
                    <div class="form-group">
                        <label>Color:</label>
                        <input type="color" id="sig-color" value="#000000" onchange="updateTypedSignature()">
                    </div>
                    <canvas id="typed-signature-canvas" width="600" height="200"></canvas>
                    <div class="button-group">
                        <button onclick="downloadTypedSignature()" class="btn-primary">
                            <i class="fas fa-download"></i> Download PNG
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    initializeCanvas();
};

function initializeCanvas() {
    signatureCanvas = document.getElementById('signature-canvas');
    signatureCtx = signatureCanvas.getContext('2d');
    
    signatureCtx.strokeStyle = '#000000';
    signatureCtx.lineWidth = 2;
    signatureCtx.lineCap = 'round';
    signatureCtx.lineJoin = 'round';

    // Mouse events
    signatureCanvas.addEventListener('mousedown', startDrawing);
    signatureCanvas.addEventListener('mousemove', draw);
    signatureCanvas.addEventListener('mouseup', stopDrawing);
    signatureCanvas.addEventListener('mouseout', stopDrawing);

    // Touch events
    signatureCanvas.addEventListener('touchstart', handleTouch);
    signatureCanvas.addEventListener('touchmove', handleTouch);
    signatureCanvas.addEventListener('touchend', stopDrawing);
}

function switchSigTab(tab) {
    document.querySelectorAll('.sig-tab-btn').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    document.getElementById('draw-sig-tab').classList.toggle('hidden', tab !== 'draw');
    document.getElementById('type-sig-tab').classList.toggle('hidden', tab !== 'type');
}

function startDrawing(e) {
    isDrawing = true;
    const rect = signatureCanvas.getBoundingClientRect();
    signatureCtx.beginPath();
    signatureCtx.moveTo(e.clientX - rect.left, e.clientY - rect.top);
}

function draw(e) {
    if (!isDrawing) return;
    const rect = signatureCanvas.getBoundingClientRect();
    signatureCtx.lineTo(e.clientX - rect.left, e.clientY - rect.top);
    signatureCtx.stroke();
}

function stopDrawing() {
    isDrawing = false;
}

function handleTouch(e) {
    e.preventDefault();
    const touch = e.touches[0];
    const mouseEvent = new MouseEvent(e.type === 'touchstart' ? 'mousedown' : 'mousemove', {
        clientX: touch.clientX,
        clientY: touch.clientY
    });
    signatureCanvas.dispatchEvent(mouseEvent);
}

function clearSignature() {
    signatureCtx.clearRect(0, 0, signatureCanvas.width, signatureCanvas.height);
}

function downloadSignature() {
    const link = document.createElement('a');
    link.download = 'signature.png';
    link.href = signatureCanvas.toDataURL();
    link.click();
}

function updateTypedSignature() {
    const name = document.getElementById('sig-name').value || 'Your Signature';
    const font = document.getElementById('sig-font').value;
    const size = document.getElementById('sig-size').value;
    const color = document.getElementById('sig-color').value;

    const canvas = document.getElementById('typed-signature-canvas');
    const ctx = canvas.getContext('2d');

    ctx.clearRect(0, 0, canvas.width, canvas.height);
    ctx.font = `${size}px ${font}`;
    ctx.fillStyle = color;
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText(name, canvas.width / 2, canvas.height / 2);
}

function downloadTypedSignature() {
    const canvas = document.getElementById('typed-signature-canvas');
    const link = document.createElement('a');
    link.download = 'signature.png';
    link.href = canvas.toDataURL();
    link.click();
}
