// Screen Recorder

let mediaRecorder = null;
let recordedChunks = [];

window.initializeScreenRecorder = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-desktop"></i> Screen Recorder</h2>
                <p>Record your screen, window, or tab</p>
            </div>

            <div class="recorder-container">
                <div class="recorder-status">
                    <div id="recorder-status-icon" class="status-icon idle">
                        <i class="fas fa-circle"></i>
                    </div>
                    <div id="recorder-status-text" class="status-text">Ready to record</div>
                    <div id="recorder-timer" class="timer hidden">00:00</div>
                </div>

                <div class="form-group">
                    <label>Record:</label>
                    <select id="record-source">
                        <option value="screen">Entire Screen</option>
                        <option value="window">Application Window</option>
                        <option value="tab">Browser Tab</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="record-audio" checked> Include system audio
                    </label>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="record-mic"> Include microphone
                    </label>
                </div>

                <div class="button-group">
                    <button id="start-record-btn" onclick="startRecording()" class="btn-primary">
                        <i class="fas fa-circle"></i> Start Recording
                    </button>
                    <button id="stop-record-btn" onclick="stopRecording()" class="btn-secondary hidden">
                        <i class="fas fa-stop"></i> Stop Recording
                    </button>
                </div>

                <div id="recording-preview" class="preview-section hidden">
                    <h3>Recording Preview:</h3>
                    <video id="preview-video" controls style="max-width: 100%;"></video>
                    <button onclick="downloadRecording()" class="btn-download">
                        <i class="fas fa-download"></i> Download Recording
                    </button>
                </div>
            </div>
        </div>
    `;
};

let startTime;
let timerInterval;

async function startRecording() {
    try {
        const displayStream = await navigator.mediaDevices.getDisplayMedia({
            video: { mediaSource: 'screen' },
            audio: document.getElementById('record-audio').checked
        });

        let tracks = displayStream.getTracks();

        if (document.getElementById('record-mic').checked) {
            const audioStream = await navigator.mediaDevices.getUserMedia({ audio: true });
            tracks = tracks.concat(audioStream.getAudioTracks());
        }

        const stream = new MediaStream(tracks);
        mediaRecorder = new MediaRecorder(stream, { mimeType: 'video/webm' });
        recordedChunks = [];

        mediaRecorder.ondataavailable = (event) => {
            if (event.data.size > 0) {
                recordedChunks.push(event.data);
            }
        };

        mediaRecorder.onstop = () => {
            const blob = new Blob(recordedChunks, { type: 'video/webm' });
            const url = URL.createObjectURL(blob);
            document.getElementById('preview-video').src = url;
            document.getElementById('recording-preview').classList.remove('hidden');
            
            stream.getTracks().forEach(track => track.stop());
            updateRecorderStatus('idle', 'Recording complete');
        };

        mediaRecorder.start();
        startTime = Date.now();
        timerInterval = setInterval(updateTimer, 1000);

        updateRecorderStatus('recording', 'Recording...');
        document.getElementById('start-record-btn').classList.add('hidden');
        document.getElementById('stop-record-btn').classList.remove('hidden');

    } catch (error) {
        alert('Error starting recording: ' + error.message);
    }
}

function stopRecording() {
    if (mediaRecorder && mediaRecorder.state !== 'inactive') {
        mediaRecorder.stop();
        clearInterval(timerInterval);
        document.getElementById('start-record-btn').classList.remove('hidden');
        document.getElementById('stop-record-btn').classList.add('hidden');
        document.getElementById('recorder-timer').classList.add('hidden');
    }
}

function updateTimer() {
    const elapsed = Math.floor((Date.now() - startTime) / 1000);
    const minutes = Math.floor(elapsed / 60).toString().padStart(2, '0');
    const seconds = (elapsed % 60).toString().padStart(2, '0');
    document.getElementById('recorder-timer').textContent = `${minutes}:${seconds}`;
    document.getElementById('recorder-timer').classList.remove('hidden');
}

function updateRecorderStatus(status, text) {
    const icon = document.getElementById('recorder-status-icon');
    icon.className = `status-icon ${status}`;
    document.getElementById('recorder-status-text').textContent = text;
}

function downloadRecording() {
    const blob = new Blob(recordedChunks, { type: 'video/webm' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `screen-recording-${Date.now()}.webm`;
    a.click();
    URL.revokeObjectURL(url);
}
