// Rotate PDF - Rotate pages with live preview
let rotatePdfFile = null;
let rotatePdfDoc = null;
let rotationAngles = {};

window.initializeRotatePdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-redo"></i> Rotate PDF Pages</h2>
                <p>Rotate PDF pages by 90°, 180°, or 270°</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="rotate-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <input type="file" id="rotate-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="rotate-controls" class="rotate-controls hidden">
                <button class="rotate-btn" onclick="rotateAllPages(90)">
                    <i class="fas fa-redo"></i> Rotate All 90°
                </button>
                <button class="rotate-btn" onclick="rotateAllPages(180)">
                    <i class="fas fa-sync"></i> Rotate All 180°
                </button>
                <button class="rotate-btn" onclick="rotateAllPages(270)">
                    <i class="fas fa-undo"></i> Rotate All 270°
                </button>
                <button class="rotate-btn" onclick="resetAllRotations()">
                    <i class="fas fa-times"></i> Reset All
                </button>
            </div>

            <div id="rotate-preview" class="hidden"></div>

            <div id="rotate-actions" class="action-buttons hidden">
                <button onclick="rotatePdfNow()" class="btn-primary">
                    <i class="fas fa-save"></i> Save Rotated PDF
                </button>
            </div>

            <div id="rotate-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('rotate-upload-zone').onclick = () => {
        document.getElementById('rotate-pdf-input').click();
    };

    document.getElementById('rotate-pdf-input').addEventListener('change', handleRotatePdfUpload);
};

async function handleRotatePdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        rotatePdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        rotatePdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        const pageCount = rotatePdfDoc.getPageCount();
        rotationAngles = {};
        for (let i = 1; i <= pageCount; i++) {
            rotationAngles[i] = 0;
        }
        
        document.getElementById('rotate-controls').classList.remove('hidden');
        await renderRotatePreview();
        document.getElementById('rotate-actions').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

async function renderRotatePreview() {
    const previewDiv = document.getElementById('rotate-preview');
    previewDiv.classList.remove('hidden');
    
    const pageCount = rotatePdfDoc.getPageCount();
    
    previewDiv.innerHTML = `
        <h3>📄 Click a page to rotate it 90° (${pageCount} pages)</h3>
        <div class="pdf-pages-grid" id="rotate-pages-grid"></div>
    `;

    const loadingTask = pdfjsLib.getDocument(await rotatePdfFile.arrayBuffer());
    const pdf = await loadingTask.promise;
    const pagesGrid = document.getElementById('rotate-pages-grid');

    for (let i = 1; i <= pageCount; i++) {
        const page = await pdf.getPage(i);
        const viewport = page.getViewport({ scale: 0.5, rotation: rotationAngles[i] });
        
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        canvas.width = viewport.width;
        canvas.height = viewport.height;

        await page.render({ canvasContext: ctx, viewport }).promise;

        const pageWrapper = document.createElement('div');
        pageWrapper.className = 'pdf-page-wrapper';
        pageWrapper.dataset.pageNum = i;
        pageWrapper.onclick = () => rotatePage(i);
        pageWrapper.style.cursor = 'pointer';

        const rotateIndicator = document.createElement('div');
        rotateIndicator.style.cssText = `
            position: absolute;
            bottom: 0.5rem;
            left: 0.5rem;
            background: rgba(99, 102, 241, 0.9);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.85rem;
        `;
        rotateIndicator.textContent = rotationAngles[i] ? `↻ ${rotationAngles[i]}°` : 'Click to rotate';

        const pageNumber = document.createElement('div');
        pageNumber.className = 'page-number-badge';
        pageNumber.textContent = `Page ${i}`;

        pageWrapper.appendChild(canvas);
        pageWrapper.appendChild(pageNumber);
        pageWrapper.appendChild(rotateIndicator);
        pagesGrid.appendChild(pageWrapper);
    }
}

function rotatePage(pageNum) {
    rotationAngles[pageNum] = (rotationAngles[pageNum] + 90) % 360;
    renderRotatePreview();
}

function rotateAllPages(angle) {
    const pageCount = rotatePdfDoc.getPageCount();
    for (let i = 1; i <= pageCount; i++) {
        rotationAngles[i] = angle % 360;
    }
    renderRotatePreview();
}

function resetAllRotations() {
    const pageCount = rotatePdfDoc.getPageCount();
    for (let i = 1; i <= pageCount; i++) {
        rotationAngles[i] = 0;
    }
    renderRotatePreview();
}

async function rotatePdfNow() {
    if (!rotatePdfDoc) {
        alert('Please upload a PDF first');
        return;
    }

    const resultDiv = document.getElementById('rotate-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Rotating PDF...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        const newPdf = await PDFLib.PDFDocument.create();
        const pageCount = rotatePdfDoc.getPageCount();

        for (let i = 1; i <= pageCount; i++) {
            const [copiedPage] = await newPdf.copyPages(rotatePdfDoc, [i - 1]);
            
            if (rotationAngles[i]) {
                const currentRotation = copiedPage.getRotation().angle;
                copiedPage.setRotation(PDFLib.degrees(currentRotation + rotationAngles[i]));
            }
            
            newPdf.addPage(copiedPage);
        }

        const pdfBytes = await newPdf.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ PDF Rotated Successfully!</h3>
                <p>${pageCount} pages processed</p>
                <div class="button-group">
                    <a href="${url}" download="rotated.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download Rotated PDF
                    </a>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Rotate error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
