// Resume Builder Tool

let resumeBuilderState = {
    data: {
        personalInfo: {},
        experience: [],
        education: [],
        skills: [],
        languages: [],
        projects: []
    },
    currentTemplate: 'modern'
};

// Initialize Resume Builder
window.initializeResumeBuilder = function() {
    const builderContainer = document.getElementById('resumeBuilder');
    if (!builderContainer) return;

    builderContainer.innerHTML = `
        <div class="resume-builder-container">
            <div class="resume-builder-header">
                <h2><i class="fas fa-user-tie"></i> Professional Resume Builder</h2>
                <p>Create stunning resumes with our easy-to-use builder</p>
            </div>

            <div class="resume-builder-main">
                <!-- Form Section -->
                <div class="resume-form-section">
                    <div class="resume-tabs">
                        <button class="tab-btn active" onclick="switchTab('personal')">Personal Info</button>
                        <button class="tab-btn" onclick="switchTab('experience')">Experience</button>
                        <button class="tab-btn" onclick="switchTab('education')">Education</button>
                        <button class="tab-btn" onclick="switchTab('skills')">Skills</button>
                        <button class="tab-btn" onclick="switchTab('projects')">Projects</button>
                    </div>

                    <div class="resume-form">
                        <!-- Personal Info Tab -->
                        <div id="tab-personal" class="tab-content active">
                            <h3>Personal Information</h3>
                            <div class="form-group">
                                <label>Full Name *</label>
                                <input type="text" id="fullName" placeholder="John Doe" onchange="updateResume()">
                            </div>
                            <div class="form-group">
                                <label>Professional Title *</label>
                                <input type="text" id="title" placeholder="Senior Software Engineer" onchange="updateResume()">
                            </div>
                            <div class="form-row">
                                <div class="form-group">
                                    <label>Email *</label>
                                    <input type="email" id="email" placeholder="john@example.com" onchange="updateResume()">
                                </div>
                                <div class="form-group">
                                    <label>Phone *</label>
                                    <input type="tel" id="phone" placeholder="+1 234 567 8900" onchange="updateResume()">
                                </div>
                            </div>
                            <div class="form-row">
                                <div class="form-group">
                                    <label>Location</label>
                                    <input type="text" id="location" placeholder="New York, USA" onchange="updateResume()">
                                </div>
                                <div class="form-group">
                                    <label>LinkedIn</label>
                                    <input type="url" id="linkedin" placeholder="linkedin.com/in/johndoe" onchange="updateResume()">
                                </div>
                            </div>
                            <div class="form-group">
                                <label>Professional Summary</label>
                                <textarea id="summary" rows="4" placeholder="Brief professional summary..." onchange="updateResume()"></textarea>
                            </div>
                        </div>

                        <!-- Experience Tab -->
                        <div id="tab-experience" class="tab-content">
                            <h3>Work Experience</h3>
                            <div id="experienceList"></div>
                            <button class="btn-add" onclick="addExperience()">
                                <i class="fas fa-plus"></i> Add Experience
                            </button>
                        </div>

                        <!-- Education Tab -->
                        <div id="tab-education" class="tab-content">
                            <h3>Education</h3>
                            <div id="educationList"></div>
                            <button class="btn-add" onclick="addEducation()">
                                <i class="fas fa-plus"></i> Add Education
                            </button>
                        </div>

                        <!-- Skills Tab -->
                        <div id="tab-skills" class="tab-content">
                            <h3>Skills</h3>
                            <div class="form-group">
                                <label>Add Skills (press Enter after each)</label>
                                <input type="text" id="skillInput" placeholder="e.g., JavaScript, React, Node.js" 
                                    onkeypress="handleSkillInput(event)">
                            </div>
                            <div id="skillsList" class="skills-display"></div>
                        </div>

                        <!-- Projects Tab -->
                        <div id="tab-projects" class="tab-content">
                            <h3>Projects</h3>
                            <div id="projectsList"></div>
                            <button class="btn-add" onclick="addProject()">
                                <i class="fas fa-plus"></i> Add Project
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Preview Section -->
                <div class="resume-preview-section">
                    <div class="preview-controls">
                        <select id="templateSelect" onchange="changeTemplate()">
                            <option value="modern">Modern</option>
                            <option value="classic">Classic</option>
                            <option value="creative">Creative</option>
                            <option value="minimal">Minimal</option>
                        </select>
                        <button class="btn-primary" onclick="downloadResume()">
                            <i class="fas fa-download"></i> Download PDF
                        </button>
                    </div>
                    <div id="resumePreview" class="resume-preview">
                        <!-- Resume preview will be rendered here -->
                    </div>
                </div>
            </div>
        </div>
    `;

    renderResume();
};

// Switch tabs
function switchTab(tabName) {
    document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
    document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
    
    event.target.classList.add('active');
    document.getElementById(`tab-${tabName}`).classList.add('active');
}

// Add experience entry
function addExperience() {
    const exp = {
        id: Date.now(),
        company: '',
        position: '',
        location: '',
        startDate: '',
        endDate: '',
        current: false,
        description: ''
    };
    resumeBuilderState.data.experience.push(exp);
    renderExperienceList();
}

// Render experience list
function renderExperienceList() {
    const container = document.getElementById('experienceList');
    container.innerHTML = resumeBuilderState.data.experience.map((exp, index) => `
        <div class="experience-item">
            <div class="item-header">
                <h4>Experience ${index + 1}</h4>
                <button class="btn-remove" onclick="removeExperience(${exp.id})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
            <div class="form-group">
                <label>Job Title</label>
                <input type="text" value="${exp.position}" 
                    onchange="updateExperience(${exp.id}, 'position', this.value)">
            </div>
            <div class="form-group">
                <label>Company</label>
                <input type="text" value="${exp.company}" 
                    onchange="updateExperience(${exp.id}, 'company', this.value)">
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Start Date</label>
                    <input type="month" value="${exp.startDate}" 
                        onchange="updateExperience(${exp.id}, 'startDate', this.value)">
                </div>
                <div class="form-group">
                    <label>End Date</label>
                    <input type="month" value="${exp.endDate}" 
                        ${exp.current ? 'disabled' : ''}
                        onchange="updateExperience(${exp.id}, 'endDate', this.value)">
                </div>
            </div>
            <div class="form-group">
                <label>
                    <input type="checkbox" ${exp.current ? 'checked' : ''}
                        onchange="updateExperience(${exp.id}, 'current', this.checked)">
                    Currently working here
                </label>
            </div>
            <div class="form-group">
                <label>Description</label>
                <textarea rows="3" onchange="updateExperience(${exp.id}, 'description', this.value)">${exp.description}</textarea>
            </div>
        </div>
    `).join('');
}

// Update experience
function updateExperience(id, field, value) {
    const exp = resumeBuilderState.data.experience.find(e => e.id === id);
    if (exp) {
        exp[field] = value;
        if (field === 'current' && value) {
            exp.endDate = '';
        }
        renderResume();
    }
}

// Remove experience
function removeExperience(id) {
    resumeBuilderState.data.experience = resumeBuilderState.data.experience.filter(e => e.id !== id);
    renderExperienceList();
    renderResume();
}

// Add education entry
function addEducation() {
    const edu = {
        id: Date.now(),
        degree: '',
        institution: '',
        location: '',
        graduationDate: '',
        gpa: ''
    };
    resumeBuilderState.data.education.push(edu);
    renderEducationList();
}

// Render education list
function renderEducationList() {
    const container = document.getElementById('educationList');
    container.innerHTML = resumeBuilderState.data.education.map((edu, index) => `
        <div class="education-item">
            <div class="item-header">
                <h4>Education ${index + 1}</h4>
                <button class="btn-remove" onclick="removeEducation(${edu.id})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
            <div class="form-group">
                <label>Degree</label>
                <input type="text" value="${edu.degree}" placeholder="Bachelor of Science in Computer Science"
                    onchange="updateEducation(${edu.id}, 'degree', this.value)">
            </div>
            <div class="form-group">
                <label>Institution</label>
                <input type="text" value="${edu.institution}" placeholder="University Name"
                    onchange="updateEducation(${edu.id}, 'institution', this.value)">
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Graduation Date</label>
                    <input type="month" value="${edu.graduationDate}"
                        onchange="updateEducation(${edu.id}, 'graduationDate', this.value)">
                </div>
                <div class="form-group">
                    <label>GPA (optional)</label>
                    <input type="text" value="${edu.gpa}" placeholder="3.8/4.0"
                        onchange="updateEducation(${edu.id}, 'gpa', this.value)">
                </div>
            </div>
        </div>
    `).join('');
}

// Update education
function updateEducation(id, field, value) {
    const edu = resumeBuilderState.data.education.find(e => e.id === id);
    if (edu) {
        edu[field] = value;
        renderResume();
    }
}

// Remove education
function removeEducation(id) {
    resumeBuilderState.data.education = resumeBuilderState.data.education.filter(e => e.id !== id);
    renderEducationList();
    renderResume();
}

// Add project
function addProject() {
    const project = {
        id: Date.now(),
        name: '',
        description: '',
        technologies: '',
        link: ''
    };
    resumeBuilderState.data.projects.push(project);
    renderProjectsList();
}

// Render projects list
function renderProjectsList() {
    const container = document.getElementById('projectsList');
    container.innerHTML = resumeBuilderState.data.projects.map((proj, index) => `
        <div class="project-item">
            <div class="item-header">
                <h4>Project ${index + 1}</h4>
                <button class="btn-remove" onclick="removeProject(${proj.id})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
            <div class="form-group">
                <label>Project Name</label>
                <input type="text" value="${proj.name}"
                    onchange="updateProject(${proj.id}, 'name', this.value)">
            </div>
            <div class="form-group">
                <label>Description</label>
                <textarea rows="2" onchange="updateProject(${proj.id}, 'description', this.value)">${proj.description}</textarea>
            </div>
            <div class="form-group">
                <label>Technologies Used</label>
                <input type="text" value="${proj.technologies}" placeholder="React, Node.js, MongoDB"
                    onchange="updateProject(${proj.id}, 'technologies', this.value)">
            </div>
        </div>
    `).join('');
}

// Update project
function updateProject(id, field, value) {
    const proj = resumeBuilderState.data.projects.find(p => p.id === id);
    if (proj) {
        proj[field] = value;
        renderResume();
    }
}

// Remove project
function removeProject(id) {
    resumeBuilderState.data.projects = resumeBuilderState.data.projects.filter(p => p.id !== id);
    renderProjectsList();
    renderResume();
}

// Handle skill input
function handleSkillInput(event) {
    if (event.key === 'Enter') {
        event.preventDefault();
        const skill = event.target.value.trim();
        if (skill) {
            resumeBuilderState.data.skills.push(skill);
            event.target.value = '';
            renderSkillsList();
            renderResume();
        }
    }
}

// Render skills list
function renderSkillsList() {
    const container = document.getElementById('skillsList');
    container.innerHTML = resumeBuilderState.data.skills.map((skill, index) => `
        <span class="skill-tag">
            ${skill}
            <i class="fas fa-times" onclick="removeSkill(${index})"></i>
        </span>
    `).join('');
}

// Remove skill
function removeSkill(index) {
    resumeBuilderState.data.skills.splice(index, 1);
    renderSkillsList();
    renderResume();
}

// Update resume from personal info
function updateResume() {
    resumeBuilderState.data.personalInfo = {
        fullName: document.getElementById('fullName')?.value || '',
        title: document.getElementById('title')?.value || '',
        email: document.getElementById('email')?.value || '',
        phone: document.getElementById('phone')?.value || '',
        location: document.getElementById('location')?.value || '',
        linkedin: document.getElementById('linkedin')?.value || '',
        summary: document.getElementById('summary')?.value || ''
    };
    renderResume();
}

// Render resume preview
function renderResume() {
    const preview = document.getElementById('resumePreview');
    if (!preview) return;

    const data = resumeBuilderState.data;
    const template = resumeBuilderState.currentTemplate;

    // Modern template
    preview.innerHTML = `
        <div class="resume-template resume-${template}">
            <!-- Header -->
            <div class="resume-header">
                <h1>${data.personalInfo.fullName || 'Your Name'}</h1>
                <p class="resume-title">${data.personalInfo.title || 'Professional Title'}</p>
                <div class="resume-contact">
                    ${data.personalInfo.email ? `<span><i class="fas fa-envelope"></i> ${data.personalInfo.email}</span>` : ''}
                    ${data.personalInfo.phone ? `<span><i class="fas fa-phone"></i> ${data.personalInfo.phone}</span>` : ''}
                    ${data.personalInfo.location ? `<span><i class="fas fa-map-marker-alt"></i> ${data.personalInfo.location}</span>` : ''}
                    ${data.personalInfo.linkedin ? `<span><i class="fab fa-linkedin"></i> ${data.personalInfo.linkedin}</span>` : ''}
                </div>
            </div>

            <!-- Summary -->
            ${data.personalInfo.summary ? `
                <div class="resume-section">
                    <h2>Professional Summary</h2>
                    <p>${data.personalInfo.summary}</p>
                </div>
            ` : ''}

            <!-- Experience -->
            ${data.experience.length > 0 ? `
                <div class="resume-section">
                    <h2>Work Experience</h2>
                    ${data.experience.map(exp => `
                        <div class="resume-item">
                            <div class="resume-item-header">
                                <h3>${exp.position || 'Position'}</h3>
                                <span class="resume-date">
                                    ${exp.startDate ? new Date(exp.startDate).toLocaleDateString('en-US', {month: 'short', year: 'numeric'}) : 'Start'} - 
                                    ${exp.current ? 'Present' : (exp.endDate ? new Date(exp.endDate).toLocaleDateString('en-US', {month: 'short', year: 'numeric'}) : 'End')}
                                </span>
                            </div>
                            <p class="resume-company">${exp.company || 'Company Name'}</p>
                            ${exp.description ? `<p class="resume-description">${exp.description}</p>` : ''}
                        </div>
                    `).join('')}
                </div>
            ` : ''}

            <!-- Education -->
            ${data.education.length > 0 ? `
                <div class="resume-section">
                    <h2>Education</h2>
                    ${data.education.map(edu => `
                        <div class="resume-item">
                            <div class="resume-item-header">
                                <h3>${edu.degree || 'Degree'}</h3>
                                <span class="resume-date">
                                    ${edu.graduationDate ? new Date(edu.graduationDate).toLocaleDateString('en-US', {month: 'short', year: 'numeric'}) : ''}
                                </span>
                            </div>
                            <p class="resume-company">${edu.institution || 'Institution'}</p>
                            ${edu.gpa ? `<p class="resume-gpa">GPA: ${edu.gpa}</p>` : ''}
                        </div>
                    `).join('')}
                </div>
            ` : ''}

            <!-- Skills -->
            ${data.skills.length > 0 ? `
                <div class="resume-section">
                    <h2>Skills</h2>
                    <div class="resume-skills">
                        ${data.skills.map(skill => `<span class="resume-skill-tag">${skill}</span>`).join('')}
                    </div>
                </div>
            ` : ''}

            <!-- Projects -->
            ${data.projects.length > 0 ? `
                <div class="resume-section">
                    <h2>Projects</h2>
                    ${data.projects.map(proj => `
                        <div class="resume-item">
                            <h3>${proj.name || 'Project Name'}</h3>
                            ${proj.description ? `<p class="resume-description">${proj.description}</p>` : ''}
                            ${proj.technologies ? `<p class="resume-technologies"><strong>Technologies:</strong> ${proj.technologies}</p>` : ''}
                        </div>
                    `).join('')}
                </div>
            ` : ''}
        </div>
    `;
}

// Change template
function changeTemplate() {
    resumeBuilderState.currentTemplate = document.getElementById('templateSelect').value;
    renderResume();
}

// Download resume as PDF
async function downloadResume() {
    const resumeElement = document.querySelector('.resume-template');
    if (!resumeElement) return;

    // Use html2pdf library
    const opt = {
        margin: 0.5,
        filename: `${resumeBuilderState.data.personalInfo.fullName || 'resume'}.pdf`,
        image: { type: 'jpeg', quality: 0.98 },
        html2canvas: { scale: 2, useCORS: true },
        jsPDF: { unit: 'in', format: 'letter', orientation: 'portrait' }
    };

    // Load html2pdf if not loaded
    if (!window.html2pdf) {
        const script = document.createElement('script');
        script.src = 'https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js';
        document.head.appendChild(script);
        script.onload = () => {
            window.html2pdf().from(resumeElement).set(opt).save();
        };
    } else {
        window.html2pdf().from(resumeElement).set(opt).save();
    }
}
