// Resize PDF - Change page size
let resizePdfFile = null;
let resizePdfDoc = null;

window.initializeResizePdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-expand-arrows-alt"></i> Resize PDF</h2>
                <p>Change PDF page size to standard formats</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="resize-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <input type="file" id="resize-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="resize-options" class="split-options hidden">
                <h3>Select Page Size</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1rem;">
                    <button class="split-mode-btn" onclick="setPageSize('A4')">
                        <i class="fas fa-file-alt"></i>
                        <div>A4</div>
                        <small>210 × 297 mm</small>
                    </button>
                    <button class="split-mode-btn" onclick="setPageSize('Letter')">
                        <i class="fas fa-file"></i>
                        <div>Letter</div>
                        <small>8.5 × 11 in</small>
                    </button>
                    <button class="split-mode-btn" onclick="setPageSize('Legal')">
                        <i class="fas fa-file-contract"></i>
                        <div>Legal</div>
                        <small>8.5 × 14 in</small>
                    </button>
                    <button class="split-mode-btn" onclick="setPageSize('A3')">
                        <i class="fas fa-file-alt"></i>
                        <div>A3</div>
                        <small>297 × 420 mm</small>
                    </button>
                    <button class="split-mode-btn" onclick="setPageSize('A5')">
                        <i class="fas fa-file"></i>
                        <div>A5</div>
                        <small>148 × 210 mm</small>
                    </button>
                    <button class="split-mode-btn" onclick="setPageSize('Tabloid')">
                        <i class="fas fa-file-alt"></i>
                        <div>Tabloid</div>
                        <small>11 × 17 in</small>
                    </button>
                </div>

                <div id="current-size" style="margin-top: 1.5rem; padding: 1rem; background: var(--bg-secondary); border-radius: 8px; text-align: center;">
                    <strong>Selected:</strong> <span id="selected-size">None</span>
                </div>
            </div>

            <div id="resize-actions" class="action-buttons hidden">
                <button onclick="resizePdfNow()" class="btn-primary">
                    <i class="fas fa-expand-arrows-alt"></i> Resize PDF
                </button>
            </div>

            <div id="resize-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('resize-upload-zone').onclick = () => {
        document.getElementById('resize-pdf-input').click();
    };

    document.getElementById('resize-pdf-input').addEventListener('change', handleResizePdfUpload);
};

let selectedPageSize = null;
const pageSizes = {
    'A4': { width: 595, height: 842 },
    'Letter': { width: 612, height: 792 },
    'Legal': { width: 612, height: 1008 },
    'A3': { width: 842, height: 1191 },
    'A5': { width: 420, height: 595 },
    'Tabloid': { width: 792, height: 1224 }
};

async function handleResizePdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        resizePdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        resizePdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        document.getElementById('resize-options').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

function setPageSize(size) {
    selectedPageSize = size;
    document.querySelectorAll('#resize-options .split-mode-btn').forEach(btn => btn.classList.remove('active'));
    event.target.closest('.split-mode-btn').classList.add('active');
    document.getElementById('selected-size').textContent = size;
    document.getElementById('resize-actions').classList.remove('hidden');
}

async function resizePdfNow() {
    if (!selectedPageSize) {
        alert('Please select a page size');
        return;
    }

    const resultDiv = document.getElementById('resize-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Resizing PDF...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        const newSize = pageSizes[selectedPageSize];
        const newPdf = await PDFLib.PDFDocument.create();
        const pages = resizePdfDoc.getPages();

        for (let i = 0; i < pages.length; i++) {
            const [embeddedPage] = await newPdf.embedPdf(resizePdfDoc, [i]);
            const page = newPdf.addPage([newSize.width, newSize.height]);
            
            // Scale to fit
            const scale = Math.min(
                newSize.width / embeddedPage.width,
                newSize.height / embeddedPage.height
            );
            
            const scaledWidth = embeddedPage.width * scale;
            const scaledHeight = embeddedPage.height * scale;
            const x = (newSize.width - scaledWidth) / 2;
            const y = (newSize.height - scaledHeight) / 2;
            
            page.drawPage(embeddedPage, {
                x, y,
                width: scaledWidth,
                height: scaledHeight
            });
        }

        const pdfBytes = await newPdf.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ PDF Resized!</h3>
                <p>All pages resized to ${selectedPageSize}</p>
                <div class="button-group">
                    <a href="${url}" download="resized.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download PDF
                    </a>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Resize error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
