// Background Remover Tool
function initializeBackgroundRemover() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="bg-remover-container">
            <h2>✂️ Remove Background</h2>
            <p class="tool-description">Remove backgrounds from images automatically</p>

            <div class="upload-zone" id="bg-upload-zone">
                <div class="upload-icon">🖼️</div>
                <h3>Drop images here or click to browse</h3>
                <p>Supports JPG, PNG, WebP, GIF</p>
                <input type="file" id="bg-file-input" accept="image/*" multiple style="display: none;">
            </div>

            <div class="preview-section" id="bg-preview-section" style="display: none;">
                <h3>Images to Process</h3>
                <div class="preview-grid" id="bg-preview-grid"></div>
                
                <div class="process-controls">
                    <button id="bg-process-btn" class="btn btn-primary">
                        <span>✂️ Remove Backgrounds</span>
                    </button>
                    <button id="bg-clear-btn" class="btn btn-secondary">Clear All</button>
                </div>
            </div>

            <div class="results-section" id="bg-results-section" style="display: none;">
                <h3>Processed Images</h3>
                <div class="results-grid" id="bg-results-grid"></div>
            </div>

            <div class="progress-container" id="bg-progress-container" style="display: none;">
                <div class="progress-bar">
                    <div class="progress-fill" id="bg-progress-fill"></div>
                </div>
                <p class="progress-text" id="bg-progress-text">Processing...</p>
            </div>
        </div>
    `;

    let selectedFiles = [];
    const uploadZone = document.getElementById('bg-upload-zone');
    const fileInput = document.getElementById('bg-file-input');
    const previewSection = document.getElementById('bg-preview-section');
    const previewGrid = document.getElementById('bg-preview-grid');
    const processBtn = document.getElementById('bg-process-btn');
    const clearBtn = document.getElementById('bg-clear-btn');
    const resultsSection = document.getElementById('bg-results-section');
    const resultsGrid = document.getElementById('bg-results-grid');
    const progressContainer = document.getElementById('bg-progress-container');
    const progressFill = document.getElementById('bg-progress-fill');
    const progressText = document.getElementById('bg-progress-text');

    // Upload zone click
    uploadZone.addEventListener('click', () => fileInput.click());

    // File input change
    fileInput.addEventListener('change', (e) => handleFiles(e.target.files));

    // Drag and drop
    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#007bff';
        uploadZone.style.background = 'rgba(0, 123, 255, 0.05)';
    });

    uploadZone.addEventListener('dragleave', () => {
        uploadZone.style.borderColor = '#ddd';
        uploadZone.style.background = 'transparent';
    });

    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#ddd';
        uploadZone.style.background = 'transparent';
        handleFiles(e.dataTransfer.files);
    });

    function handleFiles(files) {
        const imageFiles = Array.from(files).filter(file => file.type.startsWith('image/'));
        
        if (imageFiles.length === 0) {
            showNotification('Please select valid image files', 'error');
            return;
        }

        selectedFiles = [...selectedFiles, ...imageFiles];
        displayPreviews();
        previewSection.style.display = 'block';
        resultsSection.style.display = 'none';
    }

    function displayPreviews() {
        previewGrid.innerHTML = '';
        
        selectedFiles.forEach((file, index) => {
            const reader = new FileReader();
            reader.onload = (e) => {
                const previewItem = document.createElement('div');
                previewItem.className = 'preview-item';
                previewItem.innerHTML = `
                    <img src="${e.target.result}" alt="${file.name}">
                    <div class="preview-info">
                        <p class="file-name">${file.name}</p>
                        <p class="file-size">${formatFileSize(file.size)}</p>
                    </div>
                    <button class="remove-btn" data-index="${index}">×</button>
                `;
                previewGrid.appendChild(previewItem);
            };
            reader.readAsDataURL(file);
        });
    }

    // Remove file
    previewGrid.addEventListener('click', (e) => {
        if (e.target.classList.contains('remove-btn')) {
            const index = parseInt(e.target.dataset.index);
            selectedFiles.splice(index, 1);
            
            if (selectedFiles.length === 0) {
                previewSection.style.display = 'none';
            } else {
                displayPreviews();
            }
        }
    });

    // Clear all
    clearBtn.addEventListener('click', () => {
        selectedFiles = [];
        previewSection.style.display = 'none';
        resultsSection.style.display = 'none';
        fileInput.value = '';
    });

    // Process images
    processBtn.addEventListener('click', async () => {
        if (selectedFiles.length === 0) return;

        progressContainer.style.display = 'block';
        processBtn.disabled = true;
        resultsGrid.innerHTML = '';

        const formData = new FormData();
        selectedFiles.forEach(file => formData.append('files', file));

        try {
            const xhr = new XMLHttpRequest();
            
            xhr.upload.addEventListener('progress', (e) => {
                if (e.lengthComputable) {
                    const percent = (e.loaded / e.total) * 50; // Upload is 50%
                    progressFill.style.width = percent + '%';
                    progressText.textContent = `Uploading... ${Math.round(percent)}%`;
                }
            });

            xhr.addEventListener('load', () => {
                if (xhr.status === 200) {
                    const response = JSON.parse(xhr.responseText);
                    progressFill.style.width = '100%';
                    progressText.textContent = 'Complete!';
                    
                    displayResults(response.files);
                    
                    setTimeout(() => {
                        progressContainer.style.display = 'none';
                        progressFill.style.width = '0%';
                    }, 1000);
                } else {
                    throw new Error('Processing failed');
                }
            });

            xhr.addEventListener('error', () => {
                showNotification('Failed to remove backgrounds', 'error');
                progressContainer.style.display = 'none';
                processBtn.disabled = false;
            });

            xhr.open('POST', '/api/image/remove-background');
            xhr.send(formData);

            // Simulate processing progress
            let progress = 50;
            const interval = setInterval(() => {
                if (progress < 90) {
                    progress += 5;
                    progressFill.style.width = progress + '%';
                    progressText.textContent = `Processing... ${progress}%`;
                } else {
                    clearInterval(interval);
                }
            }, 200);

        } catch (error) {
            console.error('Error:', error);
            showNotification('Failed to remove backgrounds', 'error');
            progressContainer.style.display = 'none';
            processBtn.disabled = false;
        }
    });

    function displayResults(files) {
        resultsSection.style.display = 'block';
        resultsGrid.innerHTML = '';

        files.forEach(file => {
            const resultItem = document.createElement('div');
            resultItem.className = 'result-item';
            resultItem.innerHTML = `
                <img src="${file.url}" alt="${file.name}">
                <div class="result-info">
                    <p class="file-name">${file.name}</p>
                    <a href="${file.url}" download="${file.name}" class="btn btn-sm btn-primary">
                        Download
                    </a>
                </div>
            `;
            resultsGrid.appendChild(resultItem);
        });

        processBtn.disabled = false;
        showNotification('Backgrounds removed successfully!', 'success');
    }

    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }

    function showNotification(message, type) {
        // You can implement a toast notification here
        console.log(`${type}: ${message}`);
        alert(message);
    }
}


// Expose to window for router
window.initializeBackgroundRemover = initializeBackgroundRemover;
