// Regex Tester

window.initializeRegexTester = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-search"></i> Regex Tester</h2>
                <p>Test and validate regular expressions</p>
            </div>

            <div class="form-container">
                <div class="form-group">
                    <label>Regular Expression:</label>
                    <div class="regex-input-group">
                        <span class="regex-delimiter">/</span>
                        <input type="text" id="regex-pattern" class="form-input regex-pattern" placeholder="[A-Za-z]+" oninput="testRegex()">
                        <span class="regex-delimiter">/</span>
                        <input type="text" id="regex-flags" class="form-input regex-flags" placeholder="g" maxlength="5" oninput="testRegex()">
                    </div>
                    <small>Flags: g (global), i (case-insensitive), m (multiline), s (dotAll), u (unicode)</small>
                </div>

                <div class="form-group">
                    <label>Test String:</label>
                    <textarea id="regex-test-string" rows="8" class="form-input" placeholder="Enter text to test..." oninput="testRegex()"></textarea>
                </div>

                <div id="regex-results" class="results-section hidden">
                    <h3>Results:</h3>
                    <div id="regex-match-count" class="match-count"></div>
                    <div id="regex-matches" class="matches-list"></div>
                    <div id="regex-highlighted" class="highlighted-text"></div>
                </div>

                <div class="regex-examples">
                    <h4>Common Patterns:</h4>
                    <button onclick="setRegexExample('email')" class="btn-secondary">Email</button>
                    <button onclick="setRegexExample('url')" class="btn-secondary">URL</button>
                    <button onclick="setRegexExample('phone')" class="btn-secondary">Phone</button>
                    <button onclick="setRegexExample('date')" class="btn-secondary">Date</button>
                    <button onclick="setRegexExample('ip')" class="btn-secondary">IP Address</button>
                </div>
            </div>
        </div>
    `;
};

const regexExamples = {
    email: { pattern: '[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,}', flags: 'g', test: 'Contact us at support@example.com or sales@company.org' },
    url: { pattern: 'https?:\\/\\/(www\\.)?[-a-zA-Z0-9@:%._\\+~#=]{1,256}\\.[a-zA-Z0-9()]{1,6}\\b([-a-zA-Z0-9()@:%_\\+.~#?&//=]*)', flags: 'g', test: 'Visit https://www.example.com or http://test.org' },
    phone: { pattern: '\\(?\\d{3}\\)?[-.]?\\d{3}[-.]?\\d{4}', flags: 'g', test: 'Call 123-456-7890 or (555) 123-4567' },
    date: { pattern: '\\d{4}-\\d{2}-\\d{2}', flags: 'g', test: 'Events on 2024-12-25 and 2024-01-01' },
    ip: { pattern: '\\b(?:\\d{1,3}\\.){3}\\d{1,3}\\b', flags: 'g', test: 'Server at 192.168.1.1 or 10.0.0.1' }
};

function setRegexExample(type) {
    const example = regexExamples[type];
    document.getElementById('regex-pattern').value = example.pattern;
    document.getElementById('regex-flags').value = example.flags;
    document.getElementById('regex-test-string').value = example.test;
    testRegex();
}

function testRegex() {
    const pattern = document.getElementById('regex-pattern').value;
    const flags = document.getElementById('regex-flags').value;
    const testString = document.getElementById('regex-test-string').value;

    if (!pattern || !testString) {
        document.getElementById('regex-results').classList.add('hidden');
        return;
    }

    try {
        const regex = new RegExp(pattern, flags);
        const matches = [...testString.matchAll(regex)];

        document.getElementById('regex-results').classList.remove('hidden');
        
        // Display match count
        document.getElementById('regex-match-count').innerHTML = `
            <strong>${matches.length}</strong> match${matches.length !== 1 ? 'es' : ''} found
        `;

        // Display individual matches
        let matchesHtml = '<div class="matches-container">';
        matches.forEach((match, index) => {
            matchesHtml += `
                <div class="match-item">
                    <strong>Match ${index + 1}:</strong> "${match[0]}"
                    <span class="match-position">at position ${match.index}</span>
                </div>
            `;
        });
        matchesHtml += '</div>';
        document.getElementById('regex-matches').innerHTML = matchesHtml;

        // Highlight matches in text
        let highlightedText = testString;
        if (matches.length > 0) {
            const sortedMatches = matches.sort((a, b) => b.index - a.index);
            sortedMatches.forEach(match => {
                const before = highlightedText.substring(0, match.index);
                const matchText = match[0];
                const after = highlightedText.substring(match.index + matchText.length);
                highlightedText = before + `<span class="highlight">${escapeHtml(matchText)}</span>` + after;
            });
        }
        document.getElementById('regex-highlighted').innerHTML = `<pre>${highlightedText}</pre>`;

    } catch (error) {
        document.getElementById('regex-results').classList.remove('hidden');
        document.getElementById('regex-match-count').innerHTML = `<span class="error">Invalid regex: ${error.message}</span>`;
        document.getElementById('regex-matches').innerHTML = '';
        document.getElementById('regex-highlighted').innerHTML = '';
    }
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
