// QR Code Generator Tool
function initializeQRGenerator() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="qr-generator-container">
            <h2>📱 QR Code Generator</h2>
            <p class="tool-description">Create custom QR codes for URLs, text, and more</p>

            <div class="qr-form">
                <div class="form-group">
                    <label>Content Type:</label>
                    <select id="qr-type" onchange="updateQRForm()">
                        <option value="url">Website URL</option>
                        <option value="text">Plain Text</option>
                        <option value="email">Email</option>
                        <option value="phone">Phone Number</option>
                        <option value="sms">SMS</option>
                        <option value="wifi">WiFi</option>
                    </select>
                </div>

                <div id="qr-input-area">
                    <div class="form-group">
                        <label>Enter URL:</label>
                        <input type="text" id="qr-content" placeholder="https://example.com" value="">
                    </div>
                </div>

                <div class="form-group">
                    <label>Size: <span id="size-value">300</span>px</label>
                    <input type="range" id="qr-size" min="100" max="1000" value="300" step="50">
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Foreground Color:</label>
                        <input type="color" id="qr-color" value="#000000">
                    </div>
                    <div class="form-group">
                        <label>Background Color:</label>
                        <input type="color" id="qr-bg-color" value="#ffffff">
                    </div>
                </div>

                <button id="generate-qr-btn" class="btn btn-primary">Generate QR Code</button>
            </div>

            <div id="qr-result" class="qr-result hidden">
                <h3>Your QR Code</h3>
                <div id="qr-preview"></div>
                <button id="download-qr-btn" class="btn btn-secondary">Download QR Code</button>
            </div>
        </div>
    `;

    // Event listeners
    document.getElementById('qr-size').addEventListener('input', (e) => {
        document.getElementById('size-value').textContent = e.target.value;
    });

    document.getElementById('generate-qr-btn').addEventListener('click', generateQR);
}

function updateQRForm() {
    const type = document.getElementById('qr-type').value;
    const inputArea = document.getElementById('qr-input-area');
    
    let html = '';
    switch(type) {
        case 'url':
            html = `
                <div class="form-group">
                    <label>Enter URL:</label>
                    <input type="text" id="qr-content" placeholder="https://example.com">
                </div>`;
            break;
        case 'text':
            html = `
                <div class="form-group">
                    <label>Enter Text:</label>
                    <textarea id="qr-content" rows="4" placeholder="Enter your text here"></textarea>
                </div>`;
            break;
        case 'email':
            html = `
                <div class="form-group">
                    <label>Email Address:</label>
                    <input type="email" id="qr-content" placeholder="email@example.com">
                </div>`;
            break;
        case 'phone':
            html = `
                <div class="form-group">
                    <label>Phone Number:</label>
                    <input type="tel" id="qr-content" placeholder="+1234567890">
                </div>`;
            break;
        case 'sms':
            html = `
                <div class="form-group">
                    <label>Phone Number:</label>
                    <input type="tel" id="qr-phone" placeholder="+1234567890">
                </div>
                <div class="form-group">
                    <label>Message:</label>
                    <textarea id="qr-message" rows="3" placeholder="Your message"></textarea>
                </div>`;
            break;
        case 'wifi':
            html = `
                <div class="form-group">
                    <label>Network Name (SSID):</label>
                    <input type="text" id="qr-ssid" placeholder="WiFi Network Name">
                </div>
                <div class="form-group">
                    <label>Password:</label>
                    <input type="text" id="qr-password" placeholder="WiFi Password">
                </div>
                <div class="form-group">
                    <label>Encryption:</label>
                    <select id="qr-encryption">
                        <option value="WPA">WPA/WPA2</option>
                        <option value="WEP">WEP</option>
                        <option value="nopass">None</option>
                    </select>
                </div>`;
            break;
    }
    inputArea.innerHTML = html;
}

async function generateQR() {
    const type = document.getElementById('qr-type').value;
    const size = document.getElementById('qr-size').value;
    const color = document.getElementById('qr-color').value;
    const bgColor = document.getElementById('qr-bg-color').value;
    
    let content = '';
    
    switch(type) {
        case 'email':
            content = 'mailto:' + document.getElementById('qr-content').value;
            break;
        case 'phone':
            content = 'tel:' + document.getElementById('qr-content').value;
            break;
        case 'sms':
            const phone = document.getElementById('qr-phone').value;
            const message = document.getElementById('qr-message').value;
            content = `sms:${phone}?body=${encodeURIComponent(message)}`;
            break;
        case 'wifi':
            const ssid = document.getElementById('qr-ssid').value;
            const password = document.getElementById('qr-password').value;
            const encryption = document.getElementById('qr-encryption').value;
            content = `WIFI:T:${encryption};S:${ssid};P:${password};;`;
            break;
        default:
            content = document.getElementById('qr-content').value;
    }

    if (!content) {
        alert('Please enter content for the QR code');
        return;
    }

    try {
        const response = await fetch('/api/utility/qr-generate', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ text: content, size, color, bgColor })
        });

        const data = await response.json();

        if (data.success) {
            const resultDiv = document.getElementById('qr-result');
            const preview = document.getElementById('qr-preview');
            
            preview.innerHTML = `<img src="${data.url}" alt="QR Code" style="max-width: 100%;">`;
            resultDiv.classList.remove('hidden');
            
            document.getElementById('download-qr-btn').onclick = () => {
                const link = document.createElement('a');
                link.href = data.url;
                link.download = data.filename;
                link.click();
            };
        } else {
            alert('Failed to generate QR code: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while generating the QR code');
    }
}

// Expose to window for router
window.initializeQRGenerator = initializeQRGenerator;
