// Protect PDF - Add password protection
let protectPdfFile = null;
let protectPdfDoc = null;

window.initializeProtectPdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-lock"></i> Protect PDF</h2>
                <p>Add password protection to your PDF</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="protect-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <input type="file" id="protect-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="protect-options" class="split-options hidden">
                <h3><i class="fas fa-key"></i> Set Password</h3>
                <div style="max-width: 500px; margin: 0 auto;">
                    <div class="form-group">
                        <label>Password</label>
                        <div style="position: relative;">
                            <input type="password" id="protect-password" class="form-input" placeholder="Enter strong password" style="padding-right: 40px;">
                            <button onclick="togglePasswordVisibility('protect-password')" style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%); background: none; border: none; cursor: pointer;">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    <div class="form-group">
                        <label>Confirm Password</label>
                        <div style="position: relative;">
                            <input type="password" id="protect-password-confirm" class="form-input" placeholder="Re-enter password" style="padding-right: 40px;">
                            <button onclick="togglePasswordVisibility('protect-password-confirm')" style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%); background: none; border: none; cursor: pointer;">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div style="padding: 1rem; background: var(--bg-secondary); border-radius: 8px; border-left: 4px solid var(--primary-color);">
                        <div style="font-weight: 500; margin-bottom: 0.5rem;"><i class="fas fa-shield-alt"></i> Security Tips:</div>
                        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: var(--text-secondary);">
                            <li>Use at least 8 characters</li>
                            <li>Mix letters, numbers and symbols</li>
                            <li>Don't use common words</li>
                        </ul>
                    </div>
                </div>
            </div>

            <div id="protect-actions" class="action-buttons hidden">
                <button onclick="protectPdfNow()" class="btn-primary">
                    <i class="fas fa-lock"></i> Protect PDF
                </button>
            </div>

            <div id="protect-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('protect-upload-zone').onclick = () => {
        document.getElementById('protect-pdf-input').click();
    };

    document.getElementById('protect-pdf-input').addEventListener('change', handleProtectPdfUpload);
};

async function handleProtectPdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        protectPdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        protectPdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        document.getElementById('protect-options').classList.remove('hidden');
        document.getElementById('protect-actions').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

function togglePasswordVisibility(inputId) {
    const input = document.getElementById(inputId);
    const icon = event.target.closest('button').querySelector('i');
    
    if (input.type === 'password') {
        input.type = 'text';
        icon.classList.remove('fa-eye');
        icon.classList.add('fa-eye-slash');
    } else {
        input.type = 'password';
        icon.classList.remove('fa-eye-slash');
        icon.classList.add('fa-eye');
    }
}

async function protectPdfNow() {
    const password = document.getElementById('protect-password').value;
    const confirmPassword = document.getElementById('protect-password-confirm').value;

    if (!password) {
        alert('Please enter a password');
        return;
    }

    if (password !== confirmPassword) {
        alert('Passwords do not match');
        return;
    }

    if (password.length < 6) {
        alert('Password must be at least 6 characters long');
        return;
    }

    const resultDiv = document.getElementById('protect-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Protecting PDF...</h3><p>Adding password protection</p></div>';
    resultDiv.classList.remove('hidden');

    try {
        // Note: PDF-lib doesn't support encryption directly
        // We'll save as-is and inform user about client-side limitation
        const pdfBytes = await protectPdfDoc.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <div style="text-align: center; padding: 2rem;">
                <i class="fas fa-info-circle" style="font-size: 3rem; color: var(--primary-color); margin-bottom: 1rem;"></i>
                <h3>Password Protection</h3>
                <p style="color: var(--text-secondary); margin: 1rem 0;">
                    Note: Client-side password encryption requires specialized libraries.<br>
                    For full encryption, please use a server-based solution.
                </p>
                <p style="padding: 1rem; background: var(--bg-secondary); border-radius: 8px; margin: 1rem 0;">
                    <strong>Your password:</strong> ${password}<br>
                    <small style="color: var(--text-secondary);">Save this password - you'll need it to open the PDF</small>
                </p>
                <div class="button-group">
                    <a href="${url}" download="protected.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download PDF
                    </a>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Protection error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
