// Pomodoro Timer

let pomodoroInterval;
let pomodoroTime = 25 * 60; // 25 minutes in seconds
let pomodoroRunning = false;
let pomodoroMode = 'work'; // work, short-break, long-break
let completedPomodoros = 0;

window.initializePomodoroTimer = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-clock"></i> Pomodoro Timer</h2>
                <p>Productivity timer with work/break intervals</p>
            </div>

            <div class="pomodoro-container">
                <div class="pomodoro-mode-indicator">
                    <span id="pomodoro-mode-text">Work Time</span>
                </div>

                <div class="pomodoro-display" id="pomodoro-display">25:00</div>

                <div class="pomodoro-progress">
                    <div class="pomodoro-progress-bar" id="pomodoro-progress-bar"></div>
                </div>

                <div class="pomodoro-controls">
                    <button onclick="startPomodoro()" id="pomodoro-start-btn" class="btn-large btn-primary">
                        <i class="fas fa-play"></i> Start
                    </button>
                    <button onclick="pausePomodoro()" id="pomodoro-pause-btn" class="btn-large btn-secondary hidden">
                        <i class="fas fa-pause"></i> Pause
                    </button>
                    <button onclick="resetPomodoro()" class="btn-large btn-secondary">
                        <i class="fas fa-redo"></i> Reset
                    </button>
                </div>

                <div class="pomodoro-stats">
                    <div class="stat-item">
                        <div class="stat-label">Completed</div>
                        <div class="stat-value" id="completed-pomodoros">0</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-label">Current Mode</div>
                        <div class="stat-value" id="current-mode">Work</div>
                    </div>
                </div>

                <div class="pomodoro-settings">
                    <h3>Settings</h3>
                    
                    <div class="form-group">
                        <label>Work Duration (minutes):</label>
                        <input type="number" id="work-duration" value="25" min="1" max="60" class="form-input">
                    </div>

                    <div class="form-group">
                        <label>Short Break (minutes):</label>
                        <input type="number" id="short-break" value="5" min="1" max="30" class="form-input">
                    </div>

                    <div class="form-group">
                        <label>Long Break (minutes):</label>
                        <input type="number" id="long-break" value="15" min="1" max="60" class="form-input">
                    </div>

                    <div class="form-group">
                        <label>Long break after:</label>
                        <input type="number" id="long-break-interval" value="4" min="2" max="10" class="form-input"> pomodoros
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="pomodoro-sound" checked> Play sound on complete
                        </label>
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="pomodoro-notification" checked> Show browser notification
                        </label>
                    </div>
                </div>
            </div>
        </div>
    `;

    requestNotificationPermission();
};

function requestNotificationPermission() {
    if ('Notification' in window && Notification.permission === 'default') {
        Notification.requestPermission();
    }
}

function startPomodoro() {
    if (pomodoroRunning) return;

    pomodoroRunning = true;
    document.getElementById('pomodoro-start-btn').classList.add('hidden');
    document.getElementById('pomodoro-pause-btn').classList.remove('hidden');

    pomodoroInterval = setInterval(() => {
        if (pomodoroTime > 0) {
            pomodoroTime--;
            updatePomodoroDisplay();
        } else {
            pomodoroComplete();
        }
    }, 1000);
}

function pausePomodoro() {
    pomodoroRunning = false;
    clearInterval(pomodoroInterval);
    document.getElementById('pomodoro-start-btn').classList.remove('hidden');
    document.getElementById('pomodoro-pause-btn').classList.add('hidden');
}

function resetPomodoro() {
    pausePomodoro();
    pomodoroMode = 'work';
    pomodoroTime = parseInt(document.getElementById('work-duration').value) * 60;
    updatePomodoroDisplay();
    document.getElementById('pomodoro-mode-text').textContent = 'Work Time';
    document.getElementById('current-mode').textContent = 'Work';
}

function pomodoroComplete() {
    clearInterval(pomodoroInterval);
    pomodoroRunning = false;

    if (document.getElementById('pomodoro-sound').checked) {
        playSound();
    }

    if (document.getElementById('pomodoro-notification').checked) {
        showNotification();
    }

    if (pomodoroMode === 'work') {
        completedPomodoros++;
        document.getElementById('completed-pomodoros').textContent = completedPomodoros;

        const longBreakInterval = parseInt(document.getElementById('long-break-interval').value);
        if (completedPomodoros % longBreakInterval === 0) {
            startLongBreak();
        } else {
            startShortBreak();
        }
    } else {
        startWorkPeriod();
    }
}

function startWorkPeriod() {
    pomodoroMode = 'work';
    pomodoroTime = parseInt(document.getElementById('work-duration').value) * 60;
    document.getElementById('pomodoro-mode-text').textContent = 'Work Time';
    document.getElementById('current-mode').textContent = 'Work';
    document.getElementById('pomodoro-start-btn').classList.remove('hidden');
    document.getElementById('pomodoro-pause-btn').classList.add('hidden');
    updatePomodoroDisplay();
}

function startShortBreak() {
    pomodoroMode = 'short-break';
    pomodoroTime = parseInt(document.getElementById('short-break').value) * 60;
    document.getElementById('pomodoro-mode-text').textContent = 'Short Break';
    document.getElementById('current-mode').textContent = 'Break';
    document.getElementById('pomodoro-start-btn').classList.remove('hidden');
    document.getElementById('pomodoro-pause-btn').classList.add('hidden');
    updatePomodoroDisplay();
}

function startLongBreak() {
    pomodoroMode = 'long-break';
    pomodoroTime = parseInt(document.getElementById('long-break').value) * 60;
    document.getElementById('pomodoro-mode-text').textContent = 'Long Break';
    document.getElementById('current-mode').textContent = 'Break';
    document.getElementById('pomodoro-start-btn').classList.remove('hidden');
    document.getElementById('pomodoro-pause-btn').classList.add('hidden');
    updatePomodoroDisplay();
}

function updatePomodoroDisplay() {
    const minutes = Math.floor(pomodoroTime / 60);
    const seconds = pomodoroTime % 60;
    document.getElementById('pomodoro-display').textContent = 
        `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;

    // Update progress bar
    let totalTime;
    if (pomodoroMode === 'work') {
        totalTime = parseInt(document.getElementById('work-duration').value) * 60;
    } else if (pomodoroMode === 'short-break') {
        totalTime = parseInt(document.getElementById('short-break').value) * 60;
    } else {
        totalTime = parseInt(document.getElementById('long-break').value) * 60;
    }
    const progress = ((totalTime - pomodoroTime) / totalTime) * 100;
    document.getElementById('pomodoro-progress-bar').style.width = progress + '%';
}

function playSound() {
    const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBis+kNjwuWMcBjGF0fPTgjMGHm7A7+OZRQ0OVK3n77BdGAg+ldryxnUqBSl+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsIGGS36+ejUBELTKXh8K5kGgU6jtXvzH4yBidxwfDXkEAKElyx6OqlWBUIQ5zd8sFxJAUofszzxnYsBSh+zPHaizsI');
    audio.play().catch(() => {});
}

function showNotification() {
    if ('Notification' in window && Notification.permission === 'granted') {
        const title = pomodoroMode === 'work' ? 'Work session complete!' : 'Break time over!';
        const body = pomodoroMode === 'work' ? 'Time for a break!' : 'Ready to focus again?';
        new Notification(title, { body, icon: '/icon.png' });
    }
}
