// PNG to SVG Converter

let pngFiles = [];

window.initializePngToSvg = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-vector-square"></i> PNG to SVG Converter</h2>
                <p>Convert raster PNG images to vector SVG</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('png-svg-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop PNG files here or click to upload</h3>
                <p>Supports: PNG (works best with simple images)</p>
                <input type="file" id="png-svg-input" accept=".png" multiple style="display: none;">
            </div>

            <div id="png-svg-preview" class="preview-section hidden">
                <h3>Selected Files (<span id="png-svg-count">0</span>)</h3>
                <div id="png-svg-grid" class="image-grid"></div>
            </div>

            <div id="png-svg-options" class="options-section hidden">
                <h3>Conversion Options</h3>
                <div class="form-group">
                    <label>Color Tolerance: <span id="color-tolerance-value">128</span></label>
                    <input type="range" id="color-tolerance" min="0" max="255" value="128" oninput="document.getElementById('color-tolerance-value').textContent = this.value">
                    <small>Lower = more colors/detail, Higher = simpler/smaller file</small>
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" id="svg-optimize" checked> Optimize SVG output
                    </label>
                </div>
                <button onclick="convertPngToSvg()" class="btn-primary">
                    <i class="fas fa-sync"></i> Convert to SVG
                </button>
            </div>

            <div id="png-svg-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('png-svg-input').addEventListener('change', handlePngSvgUpload);
};

function handlePngSvgUpload(e) {
    pngFiles = Array.from(e.target.files);
    if (!pngFiles.length) return;

    document.getElementById('png-svg-count').textContent = pngFiles.length;
    document.getElementById('png-svg-preview').classList.remove('hidden');
    document.getElementById('png-svg-options').classList.remove('hidden');

    const grid = document.getElementById('png-svg-grid');
    grid.innerHTML = '';

    pngFiles.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <img src="${URL.createObjectURL(file)}" alt="${file.name}">
            <div class="image-name">${file.name}</div>
        `;
        grid.appendChild(item);
    });
}

async function convertPngToSvg() {
    if (!pngFiles.length) return;

    const resultsDiv = document.getElementById('png-svg-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Converting to SVG... This may take a while.</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    pngFiles.forEach(file => formData.append('files', file));
    formData.append('tolerance', document.getElementById('color-tolerance').value);
    formData.append('optimize', document.getElementById('svg-optimize').checked);

    try {
        const response = await fetch('/api/convert/png-to-svg', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayPngSvgResults(data.files);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displayPngSvgResults(files) {
    const resultsDiv = document.getElementById('png-svg-results');
    let html = '<div class="results-success"><h3>✅ Conversion Complete!</h3><div class="file-results">';

    files.forEach(file => {
        html += `
            <div class="file-result-item">
                <i class="fas fa-file-code"></i>
                <span class="file-name">${file.filename}</span>
                <span class="file-size">${file.size}</span>
                <a href="${file.url}" download="${file.filename}" class="btn-download">
                    <i class="fas fa-download"></i> Download
                </a>
            </div>
        `;
    });

    html += '</div></div>';
    resultsDiv.innerHTML = html;
}
