// Photo Retoucher / Enhancer

let photoToRetouch = null;

window.initializePhotoRetoucher = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-magic"></i> Photo Retoucher</h2>
                <p>Enhance and retouch your photos</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('retouch-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop photo here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP</p>
                <input type="file" id="retouch-input" accept="image/*" style="display: none;">
            </div>

            <div id="retouch-preview" class="preview-section hidden">
                <h3>Original Photo</h3>
                <img id="retouch-original" style="max-width: 100%;">
            </div>

            <div id="retouch-options" class="options-section hidden">
                <h3>Enhancement Options</h3>
                
                <div class="form-group">
                    <label>Brightness: <span id="brightness-value">0</span></label>
                    <input type="range" id="retouch-brightness" min="-100" max="100" value="0" oninput="document.getElementById('brightness-value').textContent = this.value">
                </div>

                <div class="form-group">
                    <label>Contrast: <span id="contrast-value">0</span></label>
                    <input type="range" id="retouch-contrast" min="-100" max="100" value="0" oninput="document.getElementById('contrast-value').textContent = this.value">
                </div>

                <div class="form-group">
                    <label>Saturation: <span id="saturation-value">0</span></label>
                    <input type="range" id="retouch-saturation" min="-100" max="100" value="0" oninput="document.getElementById('saturation-value').textContent = this.value">
                </div>

                <div class="form-group">
                    <label>Sharpness: <span id="sharpness-value">0</span></label>
                    <input type="range" id="retouch-sharpness" min="0" max="100" value="0" oninput="document.getElementById('sharpness-value').textContent = this.value">
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="retouch-auto-enhance"> Auto enhance
                    </label>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="retouch-denoise"> Reduce noise
                    </label>
                </div>

                <div class="button-group">
                    <button onclick="resetRetouchSettings()" class="btn-secondary">
                        <i class="fas fa-undo"></i> Reset
                    </button>
                    <button onclick="applyRetouch()" class="btn-primary">
                        <i class="fas fa-magic"></i> Apply Enhancements
                    </button>
                </div>
            </div>

            <div id="retouch-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('retouch-input').addEventListener('change', handleRetouchUpload);
};

function handleRetouchUpload(e) {
    photoToRetouch = e.target.files[0];
    if (!photoToRetouch) return;

    const preview = document.getElementById('retouch-original');
    preview.src = URL.createObjectURL(photoToRetouch);

    document.getElementById('retouch-preview').classList.remove('hidden');
    document.getElementById('retouch-options').classList.remove('hidden');
}

function resetRetouchSettings() {
    document.getElementById('retouch-brightness').value = 0;
    document.getElementById('retouch-contrast').value = 0;
    document.getElementById('retouch-saturation').value = 0;
    document.getElementById('retouch-sharpness').value = 0;
    document.getElementById('brightness-value').textContent = '0';
    document.getElementById('contrast-value').textContent = '0';
    document.getElementById('saturation-value').textContent = '0';
    document.getElementById('sharpness-value').textContent = '0';
    document.getElementById('retouch-auto-enhance').checked = false;
    document.getElementById('retouch-denoise').checked = false;
}

async function applyRetouch() {
    if (!photoToRetouch) return;

    const resultsDiv = document.getElementById('retouch-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Enhancing photo...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('image', photoToRetouch);
    formData.append('brightness', document.getElementById('retouch-brightness').value);
    formData.append('contrast', document.getElementById('retouch-contrast').value);
    formData.append('saturation', document.getElementById('retouch-saturation').value);
    formData.append('sharpness', document.getElementById('retouch-sharpness').value);
    formData.append('autoEnhance', document.getElementById('retouch-auto-enhance').checked);
    formData.append('denoise', document.getElementById('retouch-denoise').checked);

    try {
        const response = await fetch('/api/image/retouch', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Photo Enhanced!</h3>
                    <div class="comparison-view">
                        <div class="comparison-item">
                            <h4>Before</h4>
                            <img src="${URL.createObjectURL(photoToRetouch)}" style="max-width: 100%;">
                        </div>
                        <div class="comparison-item">
                            <h4>After</h4>
                            <img src="${data.url}" style="max-width: 100%;">
                        </div>
                    </div>
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download Enhanced Photo
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
