// PDF to Image - Convert PDF pages to any image format

let pdfFile = null;

window.initializePdfToImage = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-file-pdf"></i> PDF to Image</h2>
                <p>Convert PDF pages to JPG, PNG, WebP, or other image formats</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('pdf-to-image-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop PDF file here or click to upload</h3>
                <p>Supports all PDF files</p>
                <input type="file" id="pdf-to-image-input" accept=".pdf" style="display: none;">
            </div>

            <div id="pdf-file-info" class="file-info hidden"></div>

            <div id="pdf-to-image-options" class="options-section hidden">
                <h3>Conversion Options</h3>
                
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="image-format">
                        <option value="jpg">JPG/JPEG</option>
                        <option value="png">PNG</option>
                        <option value="webp">WebP</option>
                        <option value="gif">GIF</option>
                        <option value="bmp">BMP</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Quality:</label>
                    <div class="slider-container">
                        <input type="range" id="image-quality" min="50" max="100" value="90">
                        <span id="quality-display">90%</span>
                    </div>
                </div>

                <div class="form-group">
                    <label>DPI/Resolution:</label>
                    <select id="image-dpi">
                        <option value="72">72 DPI (Screen)</option>
                        <option value="150">150 DPI (Low)</option>
                        <option value="300">300 DPI (Standard)</option>
                        <option value="600">600 DPI (High)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Pages to Convert:</label>
                    <select id="page-selection">
                        <option value="all">All Pages</option>
                        <option value="first">First Page Only</option>
                        <option value="range">Page Range</option>
                    </select>
                </div>

                <div id="page-range-input" class="form-group hidden">
                    <label>Specify Pages (e.g., 1-3, 5, 7-9):</label>
                    <input type="text" id="pdf-page-range" class="form-input" placeholder="1-3, 5, 7-9">
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="auto-crop" checked>
                        Auto-crop white margins
                    </label>
                </div>

                <button onclick="convertPdfToImage()" class="btn-primary">
                    <i class="fas fa-exchange-alt"></i> Convert to Image
                </button>
            </div>

            <div id="pdf-to-image-result" class="results-section hidden"></div>
        </div>
    `;

    // Handle file upload
    document.getElementById('pdf-to-image-input').addEventListener('change', function(e) {
        pdfFile = e.target.files[0];
        if (pdfFile) {
            displayPdfInfo();
            document.getElementById('pdf-to-image-options').classList.remove('hidden');
        }
    });

    // Quality slider
    document.getElementById('image-quality').addEventListener('input', function(e) {
        document.getElementById('quality-display').textContent = e.target.value + '%';
    });

    // Page selection change
    document.getElementById('page-selection').addEventListener('change', function(e) {
        document.getElementById('page-range-input').classList.toggle('hidden', e.target.value !== 'range');
    });
};

function displayPdfInfo() {
    const infoDiv = document.getElementById('pdf-file-info');
    infoDiv.innerHTML = `
        <div class="file-info-content">
            <i class="fas fa-file-pdf"></i>
            <div>
                <p class="file-name">${pdfFile.name}</p>
                <p class="file-size">${(pdfFile.size / 1024 / 1024).toFixed(2)} MB</p>
            </div>
        </div>
    `;
    infoDiv.classList.remove('hidden');
}

async function convertPdfToImage() {
    if (!pdfFile) {
        alert('Please upload a PDF file');
        return;
    }

    const pageSelection = document.getElementById('page-selection').value;
    if (pageSelection === 'range') {
        const range = document.getElementById('pdf-page-range').value.trim();
        if (!range) {
            alert('Please specify page range');
            return;
        }
    }

    const resultDiv = document.getElementById('pdf-to-image-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Converting PDF to images...</div>';
    resultDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('pdf', pdfFile);
    formData.append('format', document.getElementById('image-format').value);
    formData.append('quality', document.getElementById('image-quality').value);
    formData.append('dpi', document.getElementById('image-dpi').value);
    formData.append('pages', pageSelection);
    formData.append('autoCrop', document.getElementById('auto-crop').checked);
    
    if (pageSelection === 'range') {
        formData.append('range', document.getElementById('pdf-page-range').value);
    }

    try {
        const response = await fetch('/api/convert/pdf-to-image', {
            method: 'POST',
            body: formData
        });

        if (!response.ok) {
            throw new Error('Conversion failed');
        }

        const data = await response.json();

        if (data.success && data.images && data.images.length > 0) {
            let html = `
                <h3>Successfully Converted!</h3>
                <p class="conversion-stats">${data.images.length} image(s) generated</p>
                <div class="image-gallery">
            `;
            
            data.images.forEach((imgData, index) => {
                html += `
                    <div class="image-item">
                        <div class="image-preview">
                            <img src="${imgData.url}" alt="Page ${imgData.pageNumber}">
                        </div>
                        <div class="image-details">
                            <p class="page-number">Page ${imgData.pageNumber}</p>
                            <p class="image-size">${(imgData.size / 1024).toFixed(2)} KB</p>
                        </div>
                        <a href="${imgData.url}" download="page-${imgData.pageNumber}.${document.getElementById('image-format').value}" class="btn-secondary">
                            <i class="fas fa-download"></i> Download
                        </a>
                    </div>
                `;
            });

            html += '</div>';
            resultDiv.innerHTML = html;
        } else {
            throw new Error('No images generated');
        }
    } catch (error) {
        resultDiv.innerHTML = `
            <div class="error">
                <i class="fas fa-exclamation-circle"></i>
                <p>Error: ${error.message}</p>
                <p>Please try again with a different PDF file.</p>
            </div>
        `;
    }
}

// CSS Styles
const styles = `
.file-info {
    margin: 20px 0;
    padding: 15px;
    background: #f0f4f8;
    border-left: 4px solid #3b82f6;
    border-radius: 6px;
}

.file-info-content {
    display: flex;
    align-items: center;
    gap: 15px;
}

.file-info-content i {
    font-size: 2rem;
    color: #d84315;
}

.file-name {
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 5px 0;
    word-break: break-word;
}

.file-size {
    font-size: 0.85rem;
    color: #9ca3af;
    margin: 0;
}

.slider-container {
    display: flex;
    align-items: center;
    gap: 10px;
}

.slider-container input[type="range"] {
    flex: 1;
}

.slider-container span {
    min-width: 50px;
    text-align: right;
    font-weight: 600;
    color: #3b82f6;
}

.image-gallery {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
    gap: 15px;
    margin-top: 20px;
}

.image-item {
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    overflow: hidden;
    transition: box-shadow 0.3s;
    display: flex;
    flex-direction: column;
}

.image-item:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.image-preview {
    width: 100%;
    height: 150px;
    overflow: hidden;
    background: #f3f4f6;
    display: flex;
    align-items: center;
    justify-content: center;
}

.image-preview img {
    max-width: 100%;
    max-height: 100%;
    object-fit: contain;
}

.image-details {
    padding: 10px;
    flex: 1;
}

.page-number {
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 3px 0;
    font-size: 0.9rem;
}

.image-size {
    font-size: 0.8rem;
    color: #9ca3af;
    margin: 0;
}

.image-item .btn-secondary {
    width: 100%;
    padding: 8px;
    background: #3b82f6;
    color: white;
    text-decoration: none;
    text-align: center;
    border: none;
    cursor: pointer;
    font-size: 0.85rem;
    transition: background 0.3s;
}

.image-item .btn-secondary:hover {
    background: #2563eb;
}

.conversion-stats {
    color: #059669;
    font-weight: 500;
    margin: 10px 0;
}
`;

// Inject styles
if (!document.getElementById('pdf-to-image-styles')) {
    const styleElement = document.createElement('style');
    styleElement.id = 'pdf-to-image-styles';
    styleElement.textContent = styles;
    document.head.appendChild(styleElement);
}
