// Advanced PDF Splitter Tool
function initializePDFSplitter() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="pdf-splitter-container">
            <h2>✂️ Advanced PDF Splitter</h2>
            <p class="tool-description">Split PDF by pages, ranges, or file size</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="pdf-split-input" accept=".pdf" onchange="loadPDFForSplit(this)">
                    <span>📁 Choose PDF File</span>
                </label>
            </div>

            <div id="split-options" class="split-options hidden">
                <div class="pdf-info">
                    <h3>PDF Information</h3>
                    <div id="pdf-info-display"></div>
                </div>

                <div class="form-group">
                    <label>Split Mode:</label>
                    <select id="split-mode" onchange="updateSplitMode()">
                        <option value="all">Split All Pages (1 page per file)</option>
                        <option value="range">Split by Page Range</option>
                        <option value="even-odd">Split Even/Odd Pages</option>
                        <option value="custom">Custom Page Selection</option>
                        <option value="intervals">Split Every N Pages</option>
                    </select>
                </div>

                <div id="split-mode-options"></div>

                <button class="btn btn-primary" onclick="splitPDF()">Split PDF</button>
            </div>

            <div id="split-results" class="result-section hidden">
                <h3>Split Results</h3>
                <div id="split-files-list"></div>
                <button class="btn btn-secondary" onclick="downloadAllSplits()">📦 Download All as ZIP</button>
            </div>
        </div>
    `;
}

let pdfSplitFile = null;
let pdfPageCount = 0;

async function loadPDFForSplit(input) {
    pdfSplitFile = input.files[0];
    if (!pdfSplitFile) return;

    const formData = new FormData();
    formData.append('pdf', pdfSplitFile);

    try {
        const response = await fetch('/api/pdf/get-info', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            pdfPageCount = data.pageCount;
            document.getElementById('pdf-info-display').innerHTML = `
                <p><strong>Pages:</strong> ${data.pageCount}</p>
                <p><strong>Size:</strong> ${formatFileSize(pdfSplitFile.size)}</p>
            `;
            document.getElementById('split-options').classList.remove('hidden');
            updateSplitMode();
        }
    } catch (error) {
        console.error('Error:', error);
        alert('Failed to load PDF information');
    }
}

function updateSplitMode() {
    const mode = document.getElementById('split-mode').value;
    const optionsDiv = document.getElementById('split-mode-options');
    
    switch(mode) {
        case 'range':
            optionsDiv.innerHTML = `
                <div class="form-group">
                    <label>Start Page:</label>
                    <input type="number" id="range-start" min="1" max="${pdfPageCount}" value="1">
                </div>
                <div class="form-group">
                    <label>End Page:</label>
                    <input type="number" id="range-end" min="1" max="${pdfPageCount}" value="${pdfPageCount}">
                </div>`;
            break;
        case 'custom':
            optionsDiv.innerHTML = `
                <div class="form-group">
                    <label>Page Numbers (comma-separated, e.g., 1,3,5-7):</label>
                    <input type="text" id="custom-pages" placeholder="1,3,5-7,10">
                </div>`;
            break;
        case 'intervals':
            optionsDiv.innerHTML = `
                <div class="form-group">
                    <label>Split Every N Pages:</label>
                    <input type="number" id="interval-pages" min="1" max="${pdfPageCount}" value="2">
                </div>`;
            break;
        default:
            optionsDiv.innerHTML = '';
    }
}

async function splitPDF() {
    if (!pdfSplitFile) return;

    const mode = document.getElementById('split-mode').value;
    const formData = new FormData();
    formData.append('pdf', pdfSplitFile);
    formData.append('mode', mode);

    if (mode === 'range') {
        formData.append('startPage', document.getElementById('range-start').value);
        formData.append('endPage', document.getElementById('range-end').value);
    } else if (mode === 'custom') {
        formData.append('pages', document.getElementById('custom-pages').value);
    } else if (mode === 'intervals') {
        formData.append('interval', document.getElementById('interval-pages').value);
    }

    try {
        const response = await fetch('/api/pdf/split-advanced', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displaySplitResults(data.files);
        } else {
            alert('Failed to split PDF: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while splitting PDF');
    }
}

function displaySplitResults(files) {
    const list = document.getElementById('split-files-list');
    list.innerHTML = '';

    files.forEach(file => {
        const item = document.createElement('div');
        item.className = 'file-item';
        item.innerHTML = `
            <span>${file.filename} (${file.pages} pages)</span>
            <button class="btn-small" onclick="downloadFile('${file.url}', '${file.filename}')">Download</button>
        `;
        list.appendChild(item);
    });

    document.getElementById('split-results').classList.remove('hidden');
}

function downloadFile(url, filename) {
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    link.click();
}

function downloadAllSplits() {
    // Implementation for ZIP download
    alert('Downloading all files...');
}


// Expose to window for router
window.initializePDFSplitter = initializePDFSplitter;
