// PDF Page Rotator Tool
function initializePDFRotator() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="pdf-rotator-container">
            <h2>🔄 PDF Page Rotator</h2>
            <p class="tool-description">Rotate specific pages or entire PDF documents</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="rotate-pdf-input" accept=".pdf" onchange="loadPDFForRotation(this)">
                    <span>📁 Choose PDF File</span>
                </label>
            </div>

            <div id="rotation-options" class="rotation-options hidden">
                <div class="pdf-info">
                    <h3>PDF Information</h3>
                    <div id="rotate-pdf-info"></div>
                </div>

                <div class="form-group">
                    <label>Rotation Angle:</label>
                    <div class="rotation-buttons">
                        <button class="rotation-btn" data-angle="90" onclick="setRotation(90)">
                            <span class="rotation-icon">↻</span> 90° CW
                        </button>
                        <button class="rotation-btn" data-angle="180" onclick="setRotation(180)">
                            <span class="rotation-icon">↻</span> 180°
                        </button>
                        <button class="rotation-btn active" data-angle="270" onclick="setRotation(270)">
                            <span class="rotation-icon">↺</span> 90° CCW
                        </button>
                    </div>
                </div>

                <div class="form-group">
                    <label>Apply To:</label>
                    <select id="rotation-mode" onchange="updateRotationMode()">
                        <option value="all">All Pages</option>
                        <option value="even">Even Pages Only</option>
                        <option value="odd">Odd Pages Only</option>
                        <option value="range">Page Range</option>
                        <option value="custom">Specific Pages</option>
                    </select>
                </div>

                <div id="rotation-page-selection" class="hidden">
                    <div class="form-group">
                        <label>Pages (e.g., 1,3,5-7):</label>
                        <input type="text" id="rotation-pages" placeholder="1,3,5-7">
                    </div>
                </div>

                <div id="pdf-page-preview" class="pdf-preview-grid"></div>

                <button class="btn btn-primary" onclick="rotatePDF()">Rotate PDF</button>
            </div>

            <div id="rotation-result" class="result-section hidden">
                <h3>Rotated PDF</h3>
                <div class="file-download">
                    <p id="rotated-filename"></p>
                    <button class="btn btn-secondary" onclick="downloadRotatedPDF()">💾 Download PDF</button>
                </div>
            </div>
        </div>
    `;
}

let rotatePDFFile = null;
let rotatePDFPages = 0;
let selectedRotation = 270;

async function loadPDFForRotation(input) {
    rotatePDFFile = input.files[0];
    if (!rotatePDFFile) return;

    const formData = new FormData();
    formData.append('pdf', rotatePDFFile);

    try {
        const response = await fetch('/api/pdf/get-info', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            rotatePDFPages = data.pageCount;
            document.getElementById('rotate-pdf-info').innerHTML = `
                <p><strong>Pages:</strong> ${data.pageCount}</p>
                <p><strong>Size:</strong> ${formatFileSize(rotatePDFFile.size)}</p>
            `;
            document.getElementById('rotation-options').classList.remove('hidden');
        }
    } catch (error) {
        console.error('Error:', error);
        alert('Failed to load PDF information');
    }
}

function setRotation(angle) {
    selectedRotation = angle;
    document.querySelectorAll('.rotation-btn').forEach(btn => {
        btn.classList.toggle('active', parseInt(btn.dataset.angle) === angle);
    });
}

function updateRotationMode() {
    const mode = document.getElementById('rotation-mode').value;
    const selectionDiv = document.getElementById('rotation-page-selection');
    
    if (mode === 'range' || mode === 'custom') {
        selectionDiv.classList.remove('hidden');
    } else {
        selectionDiv.classList.add('hidden');
    }
}

async function rotatePDF() {
    if (!rotatePDFFile) return;

    const formData = new FormData();
    formData.append('pdf', rotatePDFFile);
    formData.append('angle', selectedRotation);
    formData.append('mode', document.getElementById('rotation-mode').value);
    
    const pages = document.getElementById('rotation-pages')?.value;
    if (pages) {
        formData.append('pages', pages);
    }

    try {
        const response = await fetch('/api/pdf/rotate', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            document.getElementById('rotated-filename').textContent = data.filename;
            document.getElementById('rotation-result').classList.remove('hidden');
            window.rotatedPDFUrl = data.url;
        } else {
            alert('Failed to rotate PDF: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while rotating PDF');
    }
}

function downloadRotatedPDF() {
    if (!window.rotatedPDFUrl) return;
    
    const link = document.createElement('a');
    link.href = window.rotatedPDFUrl;
    link.download = document.getElementById('rotated-filename').textContent;
    link.click();
}


// Expose to window for router
window.initializePDFRotator = initializePDFRotator;
