// PDF Page Extractor

let pdfExtractFile = null;

window.initializePdfPageExtractor = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-file-export"></i> PDF Page Extractor</h2>
                <p>Extract specific pages from PDF documents</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('pdf-extract-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop PDF file here or click to upload</h3>
                <p>Select a PDF to extract pages from</p>
                <input type="file" id="pdf-extract-input" accept=".pdf" style="display: none;">
            </div>

            <div id="pdf-extract-info" class="info-section hidden">
                <p><strong>File:</strong> <span id="pdf-filename"></span></p>
                <p><strong>Total Pages:</strong> <span id="pdf-page-count"></span></p>
            </div>

            <div id="pdf-extract-options" class="options-section hidden">
                <h3>Select Pages to Extract</h3>
                <div class="form-group">
                    <label>Extract Method:</label>
                    <select id="extract-method" onchange="toggleExtractMethod()">
                        <option value="range">Page Range</option>
                        <option value="specific">Specific Pages</option>
                        <option value="odd">All Odd Pages</option>
                        <option value="even">All Even Pages</option>
                    </select>
                </div>

                <div id="range-input" class="form-group">
                    <label>From Page:</label>
                    <input type="number" id="page-from" min="1" value="1" class="form-input">
                    <label>To Page:</label>
                    <input type="number" id="page-to" min="1" value="1" class="form-input">
                </div>

                <div id="specific-input" class="form-group hidden">
                    <label>Pages (comma-separated):</label>
                    <input type="text" id="page-numbers" placeholder="e.g., 1, 3, 5-7, 10" class="form-input">
                    <small>Example: 1, 3, 5-7, 10 will extract pages 1, 3, 5, 6, 7, and 10</small>
                </div>

                <button onclick="extractPages()" class="btn-primary">
                    <i class="fas fa-file-export"></i> Extract Pages
                </button>
            </div>

            <div id="pdf-extract-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('pdf-extract-input').addEventListener('change', handlePdfExtractUpload);
};

async function handlePdfExtractUpload(e) {
    pdfExtractFile = e.target.files[0];
    if (!pdfExtractFile) return;

    document.getElementById('pdf-filename').textContent = pdfExtractFile.name;
    document.getElementById('pdf-extract-info').classList.remove('hidden');
    document.getElementById('pdf-extract-options').classList.remove('hidden');

    // Get page count
    const formData = new FormData();
    formData.append('pdf', pdfExtractFile);

    try {
        const response = await fetch('/api/pdf/get-info', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            document.getElementById('pdf-page-count').textContent = data.pageCount;
            document.getElementById('page-to').value = data.pageCount;
            document.getElementById('page-to').max = data.pageCount;
            document.getElementById('page-from').max = data.pageCount;
        }
    } catch (error) {
        console.error('Error getting PDF info:', error);
    }
}

function toggleExtractMethod() {
    const method = document.getElementById('extract-method').value;
    document.getElementById('range-input').classList.toggle('hidden', method !== 'range');
    document.getElementById('specific-input').classList.toggle('hidden', method !== 'specific');
}

async function extractPages() {
    if (!pdfExtractFile) return;

    const resultsDiv = document.getElementById('pdf-extract-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Extracting pages...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('pdf', pdfExtractFile);

    const method = document.getElementById('extract-method').value;
    formData.append('method', method);

    if (method === 'range') {
        formData.append('from', document.getElementById('page-from').value);
        formData.append('to', document.getElementById('page-to').value);
    } else if (method === 'specific') {
        formData.append('pages', document.getElementById('page-numbers').value);
    }

    try {
        const response = await fetch('/api/pdf/extract-pages', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Pages Extracted Successfully!</h3>
                    <p><strong>Extracted Pages:</strong> ${data.extractedPages}</p>
                    <p><strong>File Size:</strong> ${(data.size / 1024).toFixed(2)} KB</p>
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download PDF
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
