// PDF Live Editor - Canva-like interface for PDF editing

let pdfEditorState = {
    pdfDoc: null,
    currentPage: 0,
    totalPages: 0,
    canvas: null,
    ctx: null,
    elements: [],
    selectedElement: null,
    isDragging: false,
    scale: 1,
    pdfBytes: null,
    history: [],
    historyIndex: -1
};

// Initialize PDF Editor
window.initializePDFEditor = async function() {
    const editorContainer = document.getElementById('pdfEditor');
    if (!editorContainer) return;

    editorContainer.innerHTML = `
        <div class="pdf-editor-container">
            <div class="pdf-editor-header">
                <h2><i class="fas fa-wand-magic-sparkles"></i> PDF Live Editor</h2>
                <p>Upload a PDF and start editing like Canva</p>
            </div>

            <div class="pdf-editor-upload" id="pdfEditorUpload">
                <div class="upload-area" onclick="document.getElementById('pdfEditorFileInput').click()">
                    <i class="fas fa-file-pdf" style="font-size: 4rem; color: var(--primary-color);"></i>
                    <h3>Upload PDF to Edit</h3>
                    <p>Click or drag PDF file here</p>
                    <input type="file" id="pdfEditorFileInput" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div class="pdf-editor-workspace hidden" id="pdfEditorWorkspace">
                <!-- Toolbar -->
                <div class="pdf-editor-toolbar">
                    <div class="toolbar-group">
                        <button class="tool-btn" onclick="addTextElement()" title="Add Text">
                            <i class="fas fa-font"></i> Text
                        </button>
                        <button class="tool-btn" onclick="addImageElement()" title="Add Image">
                            <i class="fas fa-image"></i> Image
                        </button>
                        <button class="tool-btn" onclick="addShapeElement('rectangle')" title="Add Rectangle">
                            <i class="fas fa-square"></i> Rectangle
                        </button>
                        <button class="tool-btn" onclick="addShapeElement('circle')" title="Add Circle">
                            <i class="fas fa-circle"></i> Circle
                        </button>
                        <button class="tool-btn" onclick="addDrawingTool()" title="Draw">
                            <i class="fas fa-pen"></i> Draw
                        </button>
                    </div>
                    <div class="toolbar-group">
                        <button class="tool-btn" onclick="deleteSelected()" title="Delete">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                        <button class="tool-btn" onclick="undoLastAction()" title="Undo">
                            <i class="fas fa-undo"></i> Undo
                        </button>
                    </div>
                    <div class="toolbar-group">
                        <button class="tool-btn-primary" onclick="savePDF()">
                            <i class="fas fa-download"></i> Save PDF
                        </button>
                    </div>
                </div>

                <!-- Editor Area -->
                <div class="pdf-editor-main">
                    <!-- Sidebar for properties -->
                    <div class="pdf-editor-sidebar">
                        <h3>Properties</h3>
                        <div id="elementProperties">
                            <p class="text-muted">Select an element to edit</p>
                        </div>
                    </div>

                    <!-- Canvas Area -->
                    <div class="pdf-editor-canvas-area">
                        <div class="pdf-page-controls">
                            <button onclick="previousPage()" id="prevPageBtn">
                                <i class="fas fa-chevron-left"></i>
                            </button>
                            <span id="pageIndicator">Page 1 of 1</span>
                            <button onclick="nextPage()" id="nextPageBtn">
                                <i class="fas fa-chevron-right"></i>
                            </button>
                        </div>
                        <div class="pdf-canvas-container">
                            <canvas id="pdfEditorCanvas"></canvas>
                            <div id="pdfEditorOverlay"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Setup file upload
    document.getElementById('pdfEditorFileInput').addEventListener('change', handlePDFUpload);
    
    // Setup drag and drop for PDF upload
    const uploadArea = document.querySelector('.pdf-editor-upload .upload-area');
    if (uploadArea) {
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.style.borderColor = 'var(--primary-color)';
            uploadArea.style.backgroundColor = 'rgba(79, 70, 229, 0.1)';
        });
        
        uploadArea.addEventListener('dragleave', (e) => {
            e.preventDefault();
            uploadArea.style.borderColor = 'var(--border-color)';
            uploadArea.style.backgroundColor = '';
        });
        
        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.style.borderColor = 'var(--border-color)';
            uploadArea.style.backgroundColor = '';
            
            const files = e.dataTransfer.files;
            if (files.length > 0 && files[0].type === 'application/pdf') {
                const fileInput = document.getElementById('pdfEditorFileInput');
                fileInput.files = files;
                handlePDFUpload({ target: { files } });
            }
        });
    }
};

// Handle PDF upload
async function handlePDFUpload(event) {
    const file = event.target.files[0];
    if (!file) return;

    try {
        const arrayBuffer = await file.arrayBuffer();
        pdfEditorState.pdfBytes = new Uint8Array(arrayBuffer);
        
        // Load PDF with pdf-lib
        const { PDFDocument } = window['pdf-lib'];
        pdfEditorState.pdfDoc = await PDFDocument.load(arrayBuffer);
        pdfEditorState.totalPages = pdfEditorState.pdfDoc.getPageCount();
        pdfEditorState.currentPage = 0;

        // Hide upload, show workspace
        document.getElementById('pdfEditorUpload').classList.add('hidden');
        document.getElementById('pdfEditorWorkspace').classList.remove('hidden');

        // Render first page
        await renderCurrentPage();
    } catch (error) {
        console.error('Error loading PDF:', error);
        alert('Failed to load PDF. Please try again.');
    }
}

// Render current page
async function renderCurrentPage() {
    const canvas = document.getElementById('pdfEditorCanvas');
    const ctx = canvas.getContext('2d');
    pdfEditorState.canvas = canvas;
    pdfEditorState.ctx = ctx;

    // For rendering, we'll use PDF.js
    const pdfjsLib = window['pdfjs-dist/build/pdf'];
    pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

    const loadingTask = pdfjsLib.getDocument({ data: pdfEditorState.pdfBytes });
    const pdf = await loadingTask.promise;
    const page = await pdf.getPage(pdfEditorState.currentPage + 1);

    const viewport = page.getViewport({ scale: 1.5 });
    canvas.width = viewport.width;
    canvas.height = viewport.height;

    await page.render({
        canvasContext: ctx,
        viewport: viewport
    }).promise;

    // Update page indicator
    document.getElementById('pageIndicator').textContent = 
        `Page ${pdfEditorState.currentPage + 1} of ${pdfEditorState.totalPages}`;

    // Setup canvas interactions for element dragging
    setupCanvasInteractions();
    
    // Setup keyboard shortcuts
    setupKeyboardShortcuts();

    // Render elements
    renderElements();
}

// Setup keyboard shortcuts
function setupKeyboardShortcuts() {
    document.addEventListener('keydown', (e) => {
        // Delete selected element (Delete or Backspace)
        if ((e.key === 'Delete' || e.key === 'Backspace') && pdfEditorState.selectedElement) {
            e.preventDefault();
            deleteSelected();
        }
        
        // Undo (Ctrl+Z)
        if (e.ctrlKey && e.key === 'z' && !e.shiftKey) {
            e.preventDefault();
            undoLastAction();
        }
        
        // Redo (Ctrl+Shift+Z or Ctrl+Y)
        if ((e.ctrlKey && e.shiftKey && e.key === 'z') || (e.ctrlKey && e.key === 'y')) {
            e.preventDefault();
            redoAction();
        }
        
        // Copy (Ctrl+C)
        if (e.ctrlKey && e.key === 'c' && pdfEditorState.selectedElement) {
            e.preventDefault();
            copyElement();
        }
        
        // Paste (Ctrl+V)
        if (e.ctrlKey && e.key === 'v') {
            e.preventDefault();
            pasteElement();
        }
        
        // Navigate pages with arrow keys
        if (e.key === 'ArrowLeft' || e.key === 'PageUp') {
            previousPage();
        }
        if (e.key === 'ArrowRight' || e.key === 'PageDown') {
            nextPage();
        }
    });
}

// Setup canvas interactions for dragging elements
function setupCanvasInteractions() {
    const canvas = pdfEditorState.canvas;
    if (!canvas) return;
    
    let draggedElement = null;
    let offsetX = 0;
    let offsetY = 0;
    
    canvas.addEventListener('mousedown', (e) => {
        const rect = canvas.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;
        
        // Find clicked element (reverse order to get topmost)
        const currentPageElements = pdfEditorState.elements
            .filter(el => el.page === pdfEditorState.currentPage)
            .reverse();
            
        for (const element of currentPageElements) {
            if (isPointInElement(x, y, element)) {
                draggedElement = element;
                offsetX = x - element.x;
                offsetY = y - element.y;
                selectElement(element);
                pdfEditorState.isDragging = true;
                canvas.style.cursor = 'move';
                break;
            }
        }
    });
    
    canvas.addEventListener('mousemove', (e) => {
        if (!pdfEditorState.isDragging || !draggedElement) return;
        
        const rect = canvas.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;
        
        draggedElement.x = x - offsetX;
        draggedElement.y = y - offsetY;
        
        renderElements();
    });
    
    canvas.addEventListener('mouseup', () => {
        pdfEditorState.isDragging = false;
        draggedElement = null;
        canvas.style.cursor = 'default';
    });
    
    canvas.addEventListener('mouseleave', () => {
        if (pdfEditorState.isDragging) {
            pdfEditorState.isDragging = false;
            draggedElement = null;
            canvas.style.cursor = 'default';
        }
    });
}

// Check if point is inside element
function isPointInElement(x, y, element) {
    switch (element.type) {
        case 'text':
            const textWidth = pdfEditorState.ctx.measureText(element.text).width;
            return x >= element.x && x <= element.x + textWidth &&
                   y >= element.y - element.fontSize && y <= element.y;
        case 'image':
            return x >= element.x && x <= element.x + element.width &&
                   y >= element.y && y <= element.y + element.height;
        case 'shape':
            if (element.shape === 'rectangle') {
                return x >= element.x && x <= element.x + element.width &&
                       y >= element.y && y <= element.y + element.height;
            } else if (element.shape === 'circle') {
                const centerX = element.x + element.width / 2;
                const centerY = element.y + element.height / 2;
                const radius = element.width / 2;
                const distance = Math.sqrt((x - centerX) ** 2 + (y - centerY) ** 2);
                return distance <= radius;
            }
            break;
    }
    return false;
}

// Add text element
function addTextElement() {
    saveToHistory();
    const element = {
        type: 'text',
        x: 100,
        y: 100,
        text: 'New Text',
        fontSize: 24,
        color: '#000000',
        fontFamily: 'Helvetica',
        page: pdfEditorState.currentPage
    };
    pdfEditorState.elements.push(element);
    renderElements();
    selectElement(element);
}

// Add image element
function addImageElement() {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = 'image/*';
    input.onchange = async (e) => {
        const file = e.target.files[0];
        if (!file) return;

        saveToHistory();
        const reader = new FileReader();
        reader.onload = (event) => {
            const img = new Image();
            img.onload = () => {
                const element = {
                    type: 'image',
                    x: 100,
                    y: 100,
                    width: 200,
                    height: 200,
                    image: img,
                    src: event.target.result,
                    page: pdfEditorState.currentPage
                };
                pdfEditorState.elements.push(element);
                renderElements();
                selectElement(element);
            };
            img.src = event.target.result;
        };
        reader.readAsDataURL(file);
    };
    input.click();
}

// Add shape element
function addShapeElement(shape) {
    saveToHistory();
    const element = {
        type: 'shape',
        shape: shape,
        x: 100,
        y: 100,
        width: 150,
        height: 150,
        fillColor: '#4F46E5',
        strokeColor: '#000000',
        page: pdfEditorState.currentPage
    };
    pdfEditorState.elements.push(element);
    renderElements();
    selectElement(element);
}

// Render all elements
function renderElements() {
    // Re-render the page first (clear canvas and draw PDF)
    const canvas = pdfEditorState.canvas;
    const ctx = pdfEditorState.ctx;
    if (!canvas || !ctx) return;
    
    // Re-render PDF page
    renderCurrentPage();

    const overlay = document.getElementById('pdfEditorOverlay');
    overlay.innerHTML = '';

    pdfEditorState.elements
        .filter(el => el.page === pdfEditorState.currentPage)
        .forEach(element => {
            // Highlight selected element
            if (element === pdfEditorState.selectedElement) {
                ctx.save();
                ctx.strokeStyle = '#4F46E5';
                ctx.lineWidth = 2;
                ctx.setLineDash([5, 5]);
                
                let rectX, rectY, rectWidth, rectHeight;
                
                switch (element.type) {
                    case 'text':
                        const textWidth = ctx.measureText(element.text).width;
                        rectX = element.x - 2;
                        rectY = element.y - element.fontSize - 2;
                        rectWidth = textWidth + 4;
                        rectHeight = element.fontSize + 4;
                        ctx.strokeRect(rectX, rectY, rectWidth, rectHeight);
                        break;
                    case 'image':
                    case 'shape':
                        rectX = element.x - 2;
                        rectY = element.y - 2;
                        rectWidth = element.width + 4;
                        rectHeight = element.height + 4;
                        ctx.strokeRect(rectX, rectY, rectWidth, rectHeight);
                        break;
                }
                
                // Draw resize handles
                if (rectX !== undefined) {
                    drawResizeHandles(ctx, rectX, rectY, rectWidth, rectHeight);
                }
                
                ctx.restore();
            }
            
            // Draw element
            switch (element.type) {
                case 'text':
                    ctx.font = `${element.fontSize}px ${element.fontFamily}`;
                    ctx.fillStyle = element.color;
                    ctx.fillText(element.text, element.x, element.y);
                    break;
                case 'image':
                    if (element.image) {
                        ctx.drawImage(element.image, element.x, element.y, element.width, element.height);
                    }
                    break;
                case 'shape':
                    ctx.fillStyle = element.fillColor;
                    ctx.strokeStyle = element.strokeColor;
                    if (element.shape === 'rectangle') {
                        ctx.fillRect(element.x, element.y, element.width, element.height);
                        ctx.strokeRect(element.x, element.y, element.width, element.height);
                    } else if (element.shape === 'circle') {
                        ctx.beginPath();
                        ctx.arc(element.x + element.width / 2, element.y + element.height / 2, 
                               element.width / 2, 0, Math.PI * 2);
                        ctx.fill();
                        ctx.stroke();
                    }
                    break;
            }
        });
}

// Draw resize handles on selection box
function drawResizeHandles(ctx, x, y, width, height) {
    const handleSize = 8;
    ctx.save();
    ctx.fillStyle = '#FFFFFF';
    ctx.strokeStyle = '#4F46E5';
    ctx.lineWidth = 2;
    ctx.setLineDash([]);
    
    // Corner handles
    const corners = [
        [x, y], // top-left
        [x + width, y], // top-right
        [x, y + height], // bottom-left
        [x + width, y + height] // bottom-right
    ];
    
    corners.forEach(([cx, cy]) => {
        ctx.fillRect(cx - handleSize/2, cy - handleSize/2, handleSize, handleSize);
        ctx.strokeRect(cx - handleSize/2, cy - handleSize/2, handleSize, handleSize);
    });
    
    // Edge handles (for better resizing control)
    const edges = [
        [x + width/2, y], // top
        [x + width/2, y + height], // bottom
        [x, y + height/2], // left
        [x + width, y + height/2] // right
    ];
    
    edges.forEach(([cx, cy]) => {
        ctx.fillRect(cx - handleSize/2, cy - handleSize/2, handleSize, handleSize);
        ctx.strokeRect(cx - handleSize/2, cy - handleSize/2, handleSize, handleSize);
    });
    
    ctx.restore();
}

// Select element
function selectElement(element) {
    pdfEditorState.selectedElement = element;
    showProperties(element);
}

// Show properties panel
function showProperties(element) {
    const panel = document.getElementById('elementProperties');
    
    let html = '<div class="property-group">';
    
    if (element.type === 'text') {
        html += `
            <label>Text Content</label>
            <textarea id="textContent" rows="3" style="width: 100%; padding: 0.5rem; border: 1px solid var(--border-color); border-radius: 4px;"
                oninput="updateElementProperty('text', this.value)">${element.text}</textarea>
            <label>Font Size</label>
            <input type="number" value="${element.fontSize}" 
                oninput="updateElementProperty('fontSize', parseInt(this.value))"
                style="width: 100%; padding: 0.5rem;">
            <label>Font Family</label>
            <select id="fontFamily" onchange="updateElementProperty('fontFamily', this.value)"
                style="width: 100%; padding: 0.5rem;">
                <option value="Helvetica" ${element.fontFamily === 'Helvetica' ? 'selected' : ''}>Helvetica</option>
                <option value="Times" ${element.fontFamily === 'Times' ? 'selected' : ''}>Times</option>
                <option value="Courier" ${element.fontFamily === 'Courier' ? 'selected' : ''}>Courier</option>
                <option value="Arial" ${element.fontFamily === 'Arial' ? 'selected' : ''}>Arial</option>
            </select>
            <label>Color</label>
            <input type="color" value="${element.color}" 
                oninput="updateElementProperty('color', this.value)"
                style="width: 100%; padding: 0.5rem;">
        `;
    } else if (element.type === 'image') {
        html += `
            <label>Width (px)</label>
            <input type="number" value="${element.width}" 
                oninput="updateElementProperty('width', parseInt(this.value))"
                style="width: 100%; padding: 0.5rem;">
            <label>Height (px)</label>
            <input type="number" value="${element.height}" 
                oninput="updateElementProperty('height', parseInt(this.value))"
                style="width: 100%; padding: 0.5rem;">
            <label>
                <input type="checkbox" onchange="toggleAspectRatio(this.checked)">
                Lock aspect ratio
            </label>
        `;
    } else if (element.type === 'shape') {
        html += `
            <label>Fill Color</label>
            <input type="color" value="${element.fillColor}" 
                oninput="updateElementProperty('fillColor', this.value)"
                style="width: 100%; padding: 0.5rem;">
            <label>Stroke Color</label>
            <input type="color" value="${element.strokeColor}" 
                oninput="updateElementProperty('strokeColor', this.value)"
                style="width: 100%; padding: 0.5rem;">
            <label>Width (px)</label>
            <input type="number" value="${element.width}" 
                oninput="updateElementProperty('width', parseInt(this.value))"
                style="width: 100%; padding: 0.5rem;">
            <label>Height (px)</label>
            <input type="number" value="${element.height}" 
                oninput="updateElementProperty('height', parseInt(this.value))"
                style="width: 100%; padding: 0.5rem;">
        `;
    }
    
    // Common properties
    html += `
        <hr style="margin: 1rem 0; border: none; border-top: 1px solid var(--border-color);">
        <label>Position X</label>
        <input type="number" value="${Math.round(element.x)}" 
            oninput="updateElementProperty('x', parseInt(this.value))"
            style="width: 100%; padding: 0.5rem;">
        <label>Position Y</label>
        <input type="number" value="${Math.round(element.y)}" 
            oninput="updateElementProperty('y', parseInt(this.value))"
            style="width: 100%; padding: 0.5rem;">
        <button onclick="deleteSelected()" 
            style="width: 100%; padding: 0.75rem; margin-top: 1rem; background: var(--danger-color); color: white; border: none; border-radius: 6px; cursor: pointer;">
            <i class="fas fa-trash"></i> Delete Element
        </button>
    `;
    
    html += '</div>';
    panel.innerHTML = html;
}

// Update element property
function updateElementProperty(property, value) {
    if (pdfEditorState.selectedElement) {
        saveToHistory();
        pdfEditorState.selectedElement[property] = value;
        renderElements();
    }
}

// Delete selected element
function deleteSelected() {
    if (pdfEditorState.selectedElement) {
        saveToHistory();
        const index = pdfEditorState.elements.indexOf(pdfEditorState.selectedElement);
        if (index > -1) {
            pdfEditorState.elements.splice(index, 1);
            pdfEditorState.selectedElement = null;
            document.getElementById('elementProperties').innerHTML = 
                '<p class="text-muted">Select an element to edit</p>';
            renderElements();
        }
    }
}

// Save current state to history
function saveToHistory() {
    const state = JSON.parse(JSON.stringify(pdfEditorState.elements));
    pdfEditorState.history = pdfEditorState.history.slice(0, pdfEditorState.historyIndex + 1);
    pdfEditorState.history.push(state);
    pdfEditorState.historyIndex++;
    
    // Limit history to 50 states
    if (pdfEditorState.history.length > 50) {
        pdfEditorState.history.shift();
        pdfEditorState.historyIndex--;
    }
}

// Undo last action
function undoLastAction() {
    if (pdfEditorState.historyIndex > 0) {
        pdfEditorState.historyIndex--;
        pdfEditorState.elements = JSON.parse(JSON.stringify(pdfEditorState.history[pdfEditorState.historyIndex]));
        pdfEditorState.selectedElement = null;
        renderElements();
    }
}

// Redo action
function redoAction() {
    if (pdfEditorState.historyIndex < pdfEditorState.history.length - 1) {
        pdfEditorState.historyIndex++;
        pdfEditorState.elements = JSON.parse(JSON.stringify(pdfEditorState.history[pdfEditorState.historyIndex]));
        renderElements();
    }
}

let copiedElement = null;

// Copy element
function copyElement() {
    if (pdfEditorState.selectedElement) {
        copiedElement = JSON.parse(JSON.stringify(pdfEditorState.selectedElement));
    }
}

// Paste element
function pasteElement() {
    if (copiedElement) {
        saveToHistory();
        const newElement = JSON.parse(JSON.stringify(copiedElement));
        newElement.x += 20; // Offset to make it visible
        newElement.y += 20;
        newElement.page = pdfEditorState.currentPage;
        pdfEditorState.elements.push(newElement);
        selectElement(newElement);
        renderElements();
    }
}

// Save PDF with modifications
async function savePDF() {
    try {
        const { PDFDocument, rgb } = window['pdf-lib'];
        const pdfDoc = await PDFDocument.load(pdfEditorState.pdfBytes);

        // Apply all elements to PDF
        for (const element of pdfEditorState.elements) {
            const page = pdfDoc.getPage(element.page);
            const { height } = page.getSize();

            if (element.type === 'text') {
                page.drawText(element.text, {
                    x: element.x,
                    y: height - element.y,
                    size: element.fontSize,
                    color: hexToRgb(element.color)
                });
            }
            // Add more element types as needed
        }

        const pdfBytes = await pdfDoc.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);
        
        const link = document.createElement('a');
        link.href = url;
        link.download = `edited-${Date.now()}.pdf`;
        link.click();
    } catch (error) {
        console.error('Error saving PDF:', error);
        alert('Failed to save PDF');
    }
}

// Helper: Convert hex to RGB
function hexToRgb(hex) {
    const { rgb } = window['pdf-lib'];
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? rgb(
        parseInt(result[1], 16) / 255,
        parseInt(result[2], 16) / 255,
        parseInt(result[3], 16) / 255
    ) : rgb(0, 0, 0);
}

// Page navigation
function previousPage() {
    if (pdfEditorState.currentPage > 0) {
        pdfEditorState.currentPage--;
        renderCurrentPage();
    }
}

function nextPage() {
    if (pdfEditorState.currentPage < pdfEditorState.totalPages - 1) {
        pdfEditorState.currentPage++;
        renderCurrentPage();
    }
}

// Load PDF.js library
function loadPDFLibraries() {
    // pdf-lib
    const pdfLibScript = document.createElement('script');
    pdfLibScript.src = 'https://unpkg.com/pdf-lib@1.17.1/dist/pdf-lib.min.js';
    document.head.appendChild(pdfLibScript);

    // PDF.js for rendering
    const pdfjsScript = document.createElement('script');
    pdfjsScript.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js';
    document.head.appendChild(pdfjsScript);
}

// Load libraries when script loads
loadPDFLibraries();
