// PDF Compressor Tool
function initializePDFCompressor() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="pdf-compressor-container">
            <h2>🗜️ PDF Compressor</h2>
            <p class="tool-description">Reduce PDF file size while maintaining quality</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="compress-pdf-input" accept=".pdf" multiple onchange="loadPDFsForCompression(this)">
                    <span>📁 Choose PDF Files (Multiple)</span>
                </label>
            </div>

            <div id="compress-options" class="compress-options hidden">
                <div class="file-count-info">
                    <strong><span id="compress-count">0</span> PDFs selected</strong>
                </div>

                <div class="compression-presets">
                    <h3>Compression Level</h3>
                    <div class="preset-buttons">
                        <button class="preset-btn" data-level="low" onclick="selectCompression('low')">
                            <h4>📊 Low</h4>
                            <p>Best Quality</p>
                            <small>Minimal compression</small>
                        </button>
                        <button class="preset-btn active" data-level="medium" onclick="selectCompression('medium')">
                            <h4>⚖️ Medium</h4>
                            <p>Balanced</p>
                            <small>Recommended</small>
                        </button>
                        <button class="preset-btn" data-level="high" onclick="selectCompression('high')">
                            <h4>🔥 High</h4>
                            <p>Smallest Size</p>
                            <small>Maximum compression</small>
                        </button>
                    </div>
                </div>

                <div class="compression-settings">
                    <h3>Options</h3>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="compress-images" checked>
                            Compress embedded images
                        </label>
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="remove-duplicate">
                            Remove duplicate objects
                        </label>
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="compress-fonts">
                            Compress fonts
                        </label>
                    </div>
                </div>

                <button class="btn btn-primary" onclick="compressPDFs()">Compress PDFs</button>
            </div>

            <div id="compress-results" class="result-section hidden">
                <h3>Compression Results</h3>
                
                <div class="compression-summary">
                    <div class="summary-stat">
                        <h4>Original Total</h4>
                        <p id="pdf-original-size">0 MB</p>
                    </div>
                    <div class="summary-stat">
                        <h4>Compressed Total</h4>
                        <p id="pdf-compressed-size">0 MB</p>
                    </div>
                    <div class="summary-stat success">
                        <h4>Total Saved</h4>
                        <p id="pdf-saved-size">0 MB (<span id="pdf-saved-percent">0</span>%)</p>
                    </div>
                </div>

                <div id="compress-files-list" class="compress-list"></div>
                
                <button class="btn btn-secondary" onclick="downloadAllCompressed()">📦 Download All as ZIP</button>
            </div>
        </div>
    `;
}

let compressPDFFiles = [];
let compressionLevel = 'medium';

function loadPDFsForCompression(input) {
    compressPDFFiles = Array.from(input.files);
    document.getElementById('compress-count').textContent = compressPDFFiles.length;
    document.getElementById('compress-options').classList.remove('hidden');
}

function selectCompression(level) {
    compressionLevel = level;
    document.querySelectorAll('.preset-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.level === level);
    });
}

async function compressPDFs() {
    if (compressPDFFiles.length === 0) return;

    const formData = new FormData();
    compressPDFFiles.forEach(file => formData.append('pdfs', file));
    formData.append('level', compressionLevel);
    formData.append('compressImages', document.getElementById('compress-images').checked);
    formData.append('removeDuplicates', document.getElementById('remove-duplicate').checked);
    formData.append('compressFonts', document.getElementById('compress-fonts').checked);

    try {
        const response = await fetch('/api/pdf/compress', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayCompressionResults(data);
        } else {
            alert('Failed to compress PDFs: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred during PDF compression');
    }
}

function displayCompressionResults(data) {
    const originalSize = data.originalSize || 0;
    const compressedSize = data.compressedSize || 0;
    const saved = originalSize - compressedSize;
    const savedPercent = originalSize > 0 ? ((saved / originalSize) * 100).toFixed(1) : 0;

    document.getElementById('pdf-original-size').textContent = formatFileSize(originalSize);
    document.getElementById('pdf-compressed-size').textContent = formatFileSize(compressedSize);
    document.getElementById('pdf-saved-size').textContent = formatFileSize(saved);
    document.getElementById('pdf-saved-percent').textContent = savedPercent;

    const list = document.getElementById('compress-files-list');
    list.innerHTML = '';

    data.pdfs.forEach(pdf => {
        const reduction = ((pdf.reduction || 0) * 100).toFixed(1);
        
        const item = document.createElement('div');
        item.className = 'compress-item';
        item.innerHTML = `
            <div class="compress-info">
                <h4>${pdf.filename}</h4>
                <div class="size-comparison">
                    <span>Before: ${formatFileSize(pdf.originalSize)}</span>
                    <span class="arrow">→</span>
                    <span>After: ${formatFileSize(pdf.compressedSize)}</span>
                </div>
                <p class="reduction ${reduction > 20 ? 'success' : ''}">Reduced by ${reduction}%</p>
            </div>
            <button class="btn-small" onclick="downloadFile('${pdf.url}', '${pdf.filename}')">Download</button>
        `;
        list.appendChild(item);
    });

    document.getElementById('compress-results').classList.remove('hidden');
}

function downloadAllCompressed() {
    alert('Downloading all compressed PDFs...');
}


// Expose to window for router
window.initializePDFCompressor = initializePDFCompressor;
