// PDF Bookmarker

let bookmarkPdf = null;

window.initializePdfBookmarker = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-bookmark"></i> PDF Bookmarker</h2>
                <p>Add bookmarks and table of contents to PDFs</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('bookmark-pdf-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop PDF file here or click to upload</h3>
                <p>Supports: PDF</p>
                <input type="file" id="bookmark-pdf-input" accept=".pdf" style="display: none;">
            </div>

            <div id="bookmark-preview" class="preview-section hidden">
                <h3>PDF Loaded</h3>
                <div id="bookmark-file-info"></div>
            </div>

            <div id="bookmark-options" class="options-section hidden">
                <h3>Bookmarks</h3>
                
                <div id="bookmarks-list"></div>

                <div class="form-group">
                    <label>Bookmark Title:</label>
                    <input type="text" id="bookmark-title" class="form-input" placeholder="Chapter 1">
                </div>

                <div class="form-group">
                    <label>Page Number:</label>
                    <input type="number" id="bookmark-page" min="1" value="1" class="form-input">
                </div>

                <button onclick="addBookmark()" class="btn-secondary">
                    <i class="fas fa-plus"></i> Add Bookmark
                </button>

                <button onclick="applyBookmarks()" class="btn-primary">
                    <i class="fas fa-save"></i> Save PDF with Bookmarks
                </button>
            </div>

            <div id="bookmark-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('bookmark-pdf-input').addEventListener('change', handleBookmarkPdfUpload);
};

let bookmarks = [];

async function handleBookmarkPdfUpload(e) {
    bookmarkPdf = e.target.files[0];
    if (!bookmarkPdf) return;

    // Get PDF info
    const formData = new FormData();
    formData.append('file', bookmarkPdf);

    try {
        const response = await fetch('/api/pdf/get-info', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        document.getElementById('bookmark-file-info').innerHTML = `
            <p><strong>File:</strong> ${bookmarkPdf.name}</p>
            <p><strong>Pages:</strong> ${data.pageCount || 'Unknown'}</p>
            <p><strong>Size:</strong> ${(bookmarkPdf.size / 1024 / 1024).toFixed(2)} MB</p>
        `;

        document.getElementById('bookmark-page').max = data.pageCount || 999;
    } catch (error) {
        document.getElementById('bookmark-file-info').innerHTML = `
            <p><strong>File:</strong> ${bookmarkPdf.name}</p>
        `;
    }

    bookmarks = [];
    updateBookmarksList();
    document.getElementById('bookmark-preview').classList.remove('hidden');
    document.getElementById('bookmark-options').classList.remove('hidden');
}

function addBookmark() {
    const title = document.getElementById('bookmark-title').value;
    const page = parseInt(document.getElementById('bookmark-page').value);

    if (!title.trim()) {
        alert('Please enter bookmark title');
        return;
    }

    bookmarks.push({ title, page });
    updateBookmarksList();

    document.getElementById('bookmark-title').value = '';
    document.getElementById('bookmark-page').value = parseInt(document.getElementById('bookmark-page').value) + 1;
}

function updateBookmarksList() {
    const list = document.getElementById('bookmarks-list');
    
    if (bookmarks.length === 0) {
        list.innerHTML = '<p class="info-text">No bookmarks added yet</p>';
        return;
    }

    let html = '<div class="bookmarks-items">';
    bookmarks.forEach((bookmark, index) => {
        html += `
            <div class="bookmark-item">
                <span><strong>${bookmark.title}</strong> - Page ${bookmark.page}</span>
                <button onclick="removeBookmark(${index})" class="btn-remove"><i class="fas fa-times"></i></button>
            </div>
        `;
    });
    html += '</div>';
    list.innerHTML = html;
}

function removeBookmark(index) {
    bookmarks.splice(index, 1);
    updateBookmarksList();
}

async function applyBookmarks() {
    if (!bookmarkPdf) return;

    if (bookmarks.length === 0) {
        alert('Please add at least one bookmark');
        return;
    }

    const resultsDiv = document.getElementById('bookmark-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Adding bookmarks to PDF...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('file', bookmarkPdf);
    formData.append('bookmarks', JSON.stringify(bookmarks));

    try {
        const response = await fetch('/api/pdf/add-bookmarks', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Bookmarks Added!</h3>
                    <p><strong>Bookmarks:</strong> ${bookmarks.length}</p>
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download PDF
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
