// Add Page Numbers - Professional page numbering
let pageNumbersPdfFile = null;
let pageNumbersPdfDoc = null;

window.initializePageNumbers = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-hashtag"></i> Add Page Numbers</h2>
                <p>Add professional page numbers to your PDF</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="pagenums-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <input type="file" id="pagenums-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="pagenums-options" class="split-options hidden">
                <h3>Position</h3>
                <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 0.75rem; margin-bottom: 1.5rem;">
                    <button class="split-mode-btn" onclick="setPageNumPosition('top-left')">
                        <div>Top Left</div>
                    </button>
                    <button class="split-mode-btn active" onclick="setPageNumPosition('top-center')">
                        <div>Top Center</div>
                    </button>
                    <button class="split-mode-btn" onclick="setPageNumPosition('top-right')">
                        <div>Top Right</div>
                    </button>
                    <button class="split-mode-btn" onclick="setPageNumPosition('bottom-left')">
                        <div>Bottom Left</div>
                    </button>
                    <button class="split-mode-btn" onclick="setPageNumPosition('bottom-center')">
                        <div>Bottom Center</div>
                    </button>
                    <button class="split-mode-btn" onclick="setPageNumPosition('bottom-right')">
                        <div>Bottom Right</div>
                    </button>
                </div>

                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
                    <div class="form-group">
                        <label>Format</label>
                        <select id="pagenum-format" class="form-input">
                            <option value="number">1, 2, 3...</option>
                            <option value="page-number">Page 1, Page 2...</option>
                            <option value="of-total">1 of 10, 2 of 10...</option>
                            <option value="dash">- 1 -, - 2 -...</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Font Size</label>
                        <input type="number" id="pagenum-size" class="form-input" value="12" min="8" max="24">
                    </div>
                    <div class="form-group">
                        <label>Color</label>
                        <input type="color" id="pagenum-color" class="form-input" value="#000000">
                    </div>
                    <div class="form-group">
                        <label>Start From</label>
                        <input type="number" id="pagenum-start" class="form-input" value="1" min="1">
                    </div>
                </div>
            </div>

            <div id="pagenums-actions" class="action-buttons hidden">
                <button onclick="addPageNumbersNow()" class="btn-primary">
                    <i class="fas fa-hashtag"></i> Add Page Numbers
                </button>
            </div>

            <div id="pagenums-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('pagenums-upload-zone').onclick = () => {
        document.getElementById('pagenums-pdf-input').click();
    };

    document.getElementById('pagenums-pdf-input').addEventListener('change', handlePageNumsPdfUpload);
};

let pageNumPosition = 'top-center';

async function handlePageNumsPdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        pageNumbersPdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        pageNumbersPdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        document.getElementById('pagenums-options').classList.remove('hidden');
        document.getElementById('pagenums-actions').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

function setPageNumPosition(position) {
    pageNumPosition = position;
    document.querySelectorAll('#pagenums-options .split-mode-btn').forEach(btn => btn.classList.remove('active'));
    event.target.closest('.split-mode-btn').classList.add('active');
}

async function addPageNumbersNow() {
    if (!pageNumbersPdfDoc) return;

    const resultDiv = document.getElementById('pagenums-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Adding Page Numbers...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        const format = document.getElementById('pagenum-format').value;
        const fontSize = parseInt(document.getElementById('pagenum-size').value);
        const color = document.getElementById('pagenum-color').value;
        const startFrom = parseInt(document.getElementById('pagenum-start').value);
        
        // Convert hex color to RGB
        const r = parseInt(color.substr(1, 2), 16) / 255;
        const g = parseInt(color.substr(3, 2), 16) / 255;
        const b = parseInt(color.substr(5, 2), 16) / 255;

        const pages = pageNumbersPdfDoc.getPages();
        const totalPages = pages.length;

        for (let i = 0; i < pages.length; i++) {
            const page = pages[i];
            const { width, height } = page.getSize();
            const pageNum = i + startFrom;
            
            let text = '';
            if (format === 'number') text = `${pageNum}`;
            else if (format === 'page-number') text = `Page ${pageNum}`;
            else if (format === 'of-total') text = `${pageNum} of ${totalPages}`;
            else if (format === 'dash') text = `- ${pageNum} -`;

            let x, y;
            const textWidth = text.length * fontSize * 0.5;
            
            if (pageNumPosition.includes('left')) x = 40;
            else if (pageNumPosition.includes('right')) x = width - textWidth - 40;
            else x = (width - textWidth) / 2;
            
            if (pageNumPosition.includes('top')) y = height - 30;
            else y = 30;

            page.drawText(text, {
                x, y,
                size: fontSize,
                color: PDFLib.rgb(r, g, b)
            });
        }

        const pdfBytes = await pageNumbersPdfDoc.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ Page Numbers Added!</h3>
                <p>Added numbers to ${totalPages} pages</p>
                <div class="button-group">
                    <a href="${url}" download="numbered.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download PDF
                    </a>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Page numbers error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
