// Organize PDF - Reorder and Delete Pages
let organizePdfFile = null;
let organizePdfDoc = null;
let pageOrder = [];

window.initializeOrganizePdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="pdf-tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-sort"></i> Organize PDF Pages</h2>
                <p>Reorder, remove, or rearrange PDF pages</p>
            </div>
            
            <div class="upload-section">
                <div class="upload-zone" id="organize-upload-zone">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <h3>Click to select PDF file</h3>
                    <input type="file" id="organize-pdf-input" accept=".pdf" style="display: none;">
                </div>
            </div>

            <div id="organize-preview" class="hidden"></div>

            <div id="organize-actions" class="action-buttons hidden">
                <button onclick="organizePdfNow()" class="btn-primary">
                    <i class="fas fa-save"></i> Save Organized PDF
                </button>
            </div>

            <div id="organize-result" class="result-section hidden"></div>
        </div>
    `;

    document.getElementById('organize-upload-zone').onclick = () => {
        document.getElementById('organize-pdf-input').click();
    };

    document.getElementById('organize-pdf-input').addEventListener('change', handleOrganizePdfUpload);
};

async function handleOrganizePdfUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    try {
        organizePdfFile = file;
        const arrayBuffer = await file.arrayBuffer();
        organizePdfDoc = await PDFLib.PDFDocument.load(arrayBuffer);
        
        const pageCount = organizePdfDoc.getPageCount();
        pageOrder = Array.from({ length: pageCount }, (_, i) => i + 1);
        
        await renderOrganizePreview();
        document.getElementById('organize-actions').classList.remove('hidden');
    } catch (error) {
        alert('Failed to load PDF: ' + error.message);
    }
}

async function renderOrganizePreview() {
    const previewDiv = document.getElementById('organize-preview');
    previewDiv.classList.remove('hidden');
    
    const pageCount = organizePdfDoc.getPageCount();
    
    previewDiv.innerHTML = `
        <h3>📄 Drag pages to reorder, click ✕ to remove (${pageOrder.length} pages)</h3>
        <div class="pdf-pages-grid" id="organize-pages-grid"></div>
    `;

    const loadingTask = pdfjsLib.getDocument(await organizePdfFile.arrayBuffer());
    const pdf = await loadingTask.promise;
    const pagesGrid = document.getElementById('organize-pages-grid');

    for (let i = 0; i < pageOrder.length; i++) {
        const pageNum = pageOrder[i];
        const page = await pdf.getPage(pageNum);
        const viewport = page.getViewport({ scale: 0.5 });
        
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        canvas.width = viewport.width;
        canvas.height = viewport.height;

        await page.render({ canvasContext: ctx, viewport }).promise;

        const pageWrapper = document.createElement('div');
        pageWrapper.className = 'pdf-page-wrapper';
        pageWrapper.draggable = true;
        pageWrapper.dataset.pageNum = pageNum;
        pageWrapper.dataset.index = i;

        pageWrapper.innerHTML = `
            <button class="btn-icon" style="position: absolute; top: 0.5rem; left: 0.5rem; background: rgba(239, 68, 68, 0.9); color: white; z-index: 10;">
                <i class="fas fa-times"></i>
            </button>
        `;

        const pageNumber = document.createElement('div');
        pageNumber.className = 'page-number-badge';
        pageNumber.textContent = `Page ${pageNum}`;

        pageWrapper.insertBefore(canvas, pageWrapper.firstChild);
        pageWrapper.appendChild(pageNumber);

        // Remove button
        pageWrapper.querySelector('.btn-icon').onclick = (e) => {
            e.stopPropagation();
            removePage(i);
        };

        // Drag events
        pageWrapper.addEventListener('dragstart', handleOrgDragStart);
        pageWrapper.addEventListener('dragend', handleOrgDragEnd);
        pageWrapper.addEventListener('dragover', handleOrgDragOver);
        pageWrapper.addEventListener('drop', handleOrgDrop);

        pagesGrid.appendChild(pageWrapper);
    }
}

let orgDraggedElement = null;

function handleOrgDragStart(e) {
    orgDraggedElement = this;
    this.style.opacity = '0.5';
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/html', this.dataset.index);
}

function handleOrgDragEnd(e) {
    this.style.opacity = '1';
    orgDraggedElement = null;
}

function handleOrgDragOver(e) {
    if (e.preventDefault) {
        e.preventDefault();
    }
    e.dataTransfer.dropEffect = 'move';
    
    if (this === orgDraggedElement) return;
    
    const grid = document.getElementById('organize-pages-grid');
    const rect = this.getBoundingClientRect();
    const nextElement = (e.clientX - rect.left) / rect.width > 0.5 ? this.nextSibling : this;
    
    grid.insertBefore(orgDraggedElement, nextElement);
    
    return false;
}

function handleOrgDrop(e) {
    if (e.stopPropagation) {
        e.stopPropagation();
    }
    
    // Update pageOrder based on DOM
    const wrappers = document.querySelectorAll('.pdf-page-wrapper');
    pageOrder = Array.from(wrappers).map(w => parseInt(w.dataset.pageNum));
    
    // Update indices
    wrappers.forEach((wrapper, index) => {
        wrapper.dataset.index = index;
    });
    
    return false;
}

function removePage(index) {
    pageOrder.splice(index, 1);
    renderOrganizePreview();
}

async function organizePdfNow() {
    if (!organizePdfDoc || pageOrder.length === 0) {
        alert('No pages to organize');
        return;
    }

    const resultDiv = document.getElementById('organize-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i><h3>Organizing PDF...</h3></div>';
    resultDiv.classList.remove('hidden');

    try {
        const newPdf = await PDFLib.PDFDocument.create();
        
        for (const pageNum of pageOrder) {
            const [copiedPage] = await newPdf.copyPages(organizePdfDoc, [pageNum - 1]);
            newPdf.addPage(copiedPage);
        }

        const pdfBytes = await newPdf.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <div class="success-message">
                <i class="fas fa-check-circle"></i>
                <h3>✨ PDF Organized Successfully!</h3>
                <p>${pageOrder.length} pages in new order</p>
                <div class="button-group">
                    <a href="${url}" download="organized.pdf" class="btn-primary">
                        <i class="fas fa-download"></i> Download Organized PDF
                    </a>
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Organize error:', error);
        resultDiv.innerHTML = `<div class="error-message"><i class="fas fa-exclamation-circle"></i><h3>Error</h3><p>${error.message}</p></div>`;
    }
}
