// Mood Board Creator

let moodImages = [];

window.initializeMoodBoard = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-palette"></i> Mood Board Creator</h2>
                <p>Create visual mood boards</p>
            </div>

            <div class="mood-board-container">
                <div class="upload-section">
                    <div class="upload-zone" onclick="document.getElementById('mood-input').click()">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <h3>Add images to your mood board</h3>
                        <p>Upload multiple images</p>
                        <input type="file" id="mood-input" accept="image/*" multiple style="display: none;">
                    </div>

                    <div class="form-group">
                        <label>Board Title:</label>
                        <input type="text" id="mood-title" class="form-input" placeholder="My Mood Board">
                    </div>

                    <div class="form-group">
                        <label>Layout:</label>
                        <select id="mood-layout">
                            <option value="grid">Grid</option>
                            <option value="masonry">Masonry</option>
                            <option value="collage">Collage</option>
                            <option value="pinterest">Pinterest Style</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Background Color:</label>
                        <input type="color" id="mood-bg" value="#f5f5f5">
                    </div>

                    <button onclick="createMoodBoard()" class="btn-primary">
                        <i class="fas fa-magic"></i> Create Mood Board
                    </button>
                </div>

                <div class="mood-preview">
                    <h3>Preview</h3>
                    <div id="mood-board-preview" class="mood-board-area"></div>
                    <div id="mood-actions" class="hidden">
                        <button onclick="downloadMoodBoard()" class="btn-primary">
                            <i class="fas fa-download"></i> Download as Image
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    document.getElementById('mood-input').addEventListener('change', handleMoodImages);
};

function handleMoodImages(e) {
    const files = Array.from(e.target.files);
    files.forEach(file => {
        const reader = new FileReader();
        reader.onload = (event) => {
            moodImages.push(event.target.result);
            displayMoodImages();
        };
        reader.readAsDataURL(file);
    });
}

function displayMoodImages() {
    const preview = document.getElementById('mood-board-preview');
    const layout = document.getElementById('mood-layout').value;
    const bg = document.getElementById('mood-bg').value;
    
    preview.style.backgroundColor = bg;
    preview.innerHTML = moodImages.map((img, index) => `
        <div class="mood-image-item">
            <img src="${img}" alt="Image ${index + 1}">
            <button onclick="removeMoodImage(${index})" class="remove-btn">×</button>
        </div>
    `).join('');
    
    preview.className = 'mood-board-area layout-' + layout;
    
    if (moodImages.length > 0) {
        document.getElementById('mood-actions').classList.remove('hidden');
    }
}

function removeMoodImage(index) {
    moodImages.splice(index, 1);
    displayMoodImages();
}

function createMoodBoard() {
    if (moodImages.length === 0) {
        alert('Please add at least one image');
        return;
    }
    displayMoodImages();
}

async function downloadMoodBoard() {
    const preview = document.getElementById('mood-board-preview');
    
    // Use html2canvas library for better results (would need to be loaded)
    // For now, create a canvas with the images
    const canvas = document.createElement('canvas');
    canvas.width = 1200;
    canvas.height = 800;
    const ctx = canvas.getContext('2d');
    
    // Background
    ctx.fillStyle = document.getElementById('mood-bg').value;
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    
    // Title
    const title = document.getElementById('mood-title').value || 'Mood Board';
    ctx.fillStyle = '#333';
    ctx.font = 'bold 48px Arial';
    ctx.textAlign = 'center';
    ctx.fillText(title, canvas.width / 2, 60);
    
    // Images in grid
    const cols = Math.ceil(Math.sqrt(moodImages.length));
    const rows = Math.ceil(moodImages.length / cols);
    const imgWidth = (canvas.width - 80) / cols;
    const imgHeight = (canvas.height - 140) / rows;
    
    let loadedCount = 0;
    moodImages.forEach((imgSrc, index) => {
        const img = new Image();
        img.onload = () => {
            const col = index % cols;
            const row = Math.floor(index / cols);
            const x = 40 + col * imgWidth;
            const y = 100 + row * imgHeight;
            
            ctx.drawImage(img, x, y, imgWidth - 10, imgHeight - 10);
            
            loadedCount++;
            if (loadedCount === moodImages.length) {
                const link = document.createElement('a');
                link.download = 'mood-board.png';
                link.href = canvas.toDataURL();
                link.click();
            }
        };
        img.src = imgSrc;
    });
}
