// Mind Map Creator

let mindMapNodes = [];
let selectedNode = null;
let draggedNode = null;

window.initializeMindMap = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-container">
            <div class="tool-header">
                <h2><i class="fas fa-project-diagram"></i> Mind Map Creator</h2>
                <p>Create visual mind maps and diagrams</p>
            </div>

            <div class="mindmap-container">
                <div class="mindmap-controls">
                    <h3>Controls</h3>
                    
                    <div class="form-group">
                        <label>Central Topic:</label>
                        <input type="text" id="central-topic" class="form-input" placeholder="Main Idea" value="Main Idea">
                    </div>

                    <div class="form-group">
                        <label>Add Node:</label>
                        <input type="text" id="node-text" class="form-input" placeholder="Enter node text">
                        <button onclick="addMindMapNode()" class="btn-secondary">
                            <i class="fas fa-plus"></i> Add Node
                        </button>
                    </div>

                    <div class="form-group">
                        <label>Node Color:</label>
                        <input type="color" id="node-color" value="#3b82f6">
                    </div>

                    <div class="form-group">
                        <label>Layout Style:</label>
                        <select id="mindmap-layout">
                            <option value="radial">Radial</option>
                            <option value="tree">Tree</option>
                            <option value="organic">Organic</option>
                        </select>
                    </div>

                    <div class="button-group">
                        <button onclick="clearMindMap()" class="btn-danger">
                            <i class="fas fa-trash"></i> Clear
                        </button>
                        <button onclick="downloadMindMap()" class="btn-primary">
                            <i class="fas fa-download"></i> Download
                        </button>
                    </div>
                </div>

                <div class="mindmap-canvas-container">
                    <canvas id="mindmap-canvas" width="800" height="600"></canvas>
                </div>
            </div>
        </div>
    `;

    // Initialize with central node
    mindMapNodes = [{
        text: 'Main Idea',
        x: 400,
        y: 300,
        color: '#3b82f6',
        isCentral: true,
        children: []
    }];

    drawMindMap();
};

function addMindMapNode() {
    const text = document.getElementById('node-text').value;
    if (!text) return;

    const color = document.getElementById('node-color').value;
    
    mindMapNodes.push({
        text: text,
        x: 400 + Math.random() * 200 - 100,
        y: 300 + Math.random() * 200 - 100,
        color: color,
        isCentral: false
    });

    document.getElementById('node-text').value = '';
    drawMindMap();
}

function drawMindMap() {
    const canvas = document.getElementById('mindmap-canvas');
    const ctx = canvas.getContext('2d');
    
    // Clear canvas
    ctx.fillStyle = '#f9fafb';
    ctx.fillRect(0, 0, canvas.width, canvas.height);

    // Update central node text
    mindMapNodes[0].text = document.getElementById('central-topic').value || 'Main Idea';

    // Draw connections
    ctx.strokeStyle = '#d1d5db';
    ctx.lineWidth = 2;
    
    mindMapNodes.forEach((node, i) => {
        if (i > 0) {
            ctx.beginPath();
            ctx.moveTo(mindMapNodes[0].x, mindMapNodes[0].y);
            ctx.lineTo(node.x, node.y);
            ctx.stroke();
        }
    });

    // Draw nodes
    mindMapNodes.forEach(node => {
        // Node circle
        ctx.beginPath();
        ctx.arc(node.x, node.y, node.isCentral ? 60 : 40, 0, 2 * Math.PI);
        ctx.fillStyle = node.color;
        ctx.fill();
        ctx.strokeStyle = '#fff';
        ctx.lineWidth = 3;
        ctx.stroke();

        // Node text
        ctx.fillStyle = '#fff';
        ctx.font = node.isCentral ? 'bold 16px Arial' : '14px Arial';
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        
        // Wrap text
        const maxWidth = node.isCentral ? 100 : 70;
        const words = node.text.split(' ');
        let line = '';
        let y = node.y;
        
        words.forEach((word, i) => {
            const testLine = line + word + ' ';
            const metrics = ctx.measureText(testLine);
            if (metrics.width > maxWidth && i > 0) {
                ctx.fillText(line, node.x, y);
                line = word + ' ';
                y += 20;
            } else {
                line = testLine;
            }
        });
        ctx.fillText(line, node.x, y);
    });
}

function clearMindMap() {
    if (confirm('Clear the entire mind map?')) {
        mindMapNodes = [{
            text: 'Main Idea',
            x: 400,
            y: 300,
            color: '#3b82f6',
            isCentral: true
        }];
        drawMindMap();
    }
}

function downloadMindMap() {
    const canvas = document.getElementById('mindmap-canvas');
    const link = document.createElement('a');
    link.download = 'mind-map.png';
    link.href = canvas.toDataURL();
    link.click();
}
