// Merge PDF Tool - Simple and Clean Implementation
let mergePdfState = {
    files: [],
    docs: [],
    pdflibLoaded: false
};

// Load PDFLib
async function loadPDFLib() {
    if (mergePdfState.pdflibLoaded) {
        return;
    }

    return new Promise((resolve, reject) => {
        if (window.PDFLib) {
            mergePdfState.pdflibLoaded = true;
            resolve();
            return;
        }

        const script = document.createElement('script');
        script.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf-lib/1.17.1/pdf-lib.min.js';
        script.async = true;
        script.onload = () => {
            mergePdfState.pdflibLoaded = true;
            resolve();
        };
        script.onerror = () => {
            reject(new Error('Failed to load PDFLib'));
        };
        document.head.appendChild(script);
    });
}

// Load PDF.js for thumbnail rendering
async function loadPDFJS() {
    if (window.pdfjsLib) {
        return;
    }
    
    return new Promise((resolve, reject) => {
        const script = document.createElement('script');
        script.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js';
        script.async = true;
        script.onload = () => {
            pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';
            resolve();
        };
        script.onerror = () => reject(new Error('Failed to load PDF.js'));
        document.head.appendChild(script);
    });
}

// Render PDF thumbnail with page navigation (optimized for speed)
async function renderPdfThumbnail(file, pageNum = 1, isPopup = false) {
    try {
        await loadPDFJS();
        const arrayBuffer = await file.arrayBuffer();
        const pdf = await pdfjsLib.getDocument({ data: arrayBuffer }).promise;
        const page = await pdf.getPage(pageNum);
        
        const canvas = document.createElement('canvas');
        const scale = isPopup ? 1.2 : 0.6; // Higher quality for popup
        const viewport = page.getViewport({ scale });
        
        canvas.width = viewport.width;
        canvas.height = viewport.height;
        canvas.style.width = '100%';
        canvas.style.height = '100%';
        canvas.style.objectFit = 'contain';
        canvas.style.borderRadius = '8px';
        
        const context = canvas.getContext('2d', { alpha: false }); // Faster rendering
        await page.render({ 
            canvasContext: context, 
            viewport,
            intent: 'display'
        }).promise;
        
        return { canvas, totalPages: pdf.numPages, pdf };
    } catch (error) {
        console.error('PDF thumbnail error:', error);
        return null;
    }
}

window.initializeMergePdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="tool-page-layout">
            <!-- Left Ad Space (hidden) -->
            <!-- <div class="ad-space-left">
                <div class="demo-ad-box" style="width: 300px; height: 600px;">
                    <p>Flexible Ad</p>
                    <span>Any Size</span>
                </div>
            </div> -->

            <!-- Main Content -->
            <div class="pdf-tool-container">
                <div class="tool-header">
                    <h2><i class="fas fa-link"></i> Merge PDF</h2>
                    <p>Combine multiple PDF files into one seamless document</p>
                </div>
                
                <div class="upload-section">
                    <button onclick="event.stopPropagation(); window.history.back()" class="upload-back-btn" title="Go Back">
                        <i class="fas fa-arrow-left"></i>
                    </button>
                    <button onclick="event.stopPropagation(); clearMergePdfs()" class="upload-clear-btn hidden" id="merge-clear-btn" title="Clear all files">
                        <span class="cross-icon"></span>
                    </button>
                    <button onclick="event.stopPropagation(); mergePdfsNow()" class="upload-merge-btn hidden" id="merge-now-btn" title="Merge PDFs">
                        Download
                    </button>
                    
                    <button class="mobile-upload-btn" id="mobile-upload-btn" onclick="document.getElementById('merge-pdf-input').click()">
                        Upload or Drag and Drop Files
                    </button>
                    
                    <div class="upload-zone-wrapper" id="merge-upload-wrapper">
                        <div class="upload-zone-header" id="merge-upload-zone">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <h3>Click to select PDF files</h3>
                        <p>Upload multiple PDFs to merge or drag to reorder</p>
                        <input type="file" id="merge-pdf-input" accept=".pdf" multiple style="display: none;">
                    </div>
                    <div id="merge-file-list" class="merge-file-list-inline hidden"></div>
                    <button onclick="event.stopPropagation(); mergePdfsNow()" class="upload-process-btn hidden" id="merge-process-btn" title="Merge PDFs">
                        <i class="fas fa-layer-group"></i>
                        <span class="btn-text">Merge Now</span>
                    </button>
                </div>
            </div>

                <!-- Bottom Ad Space (hidden) -->
                <!-- <div class="ad-space-bottom">
                    <div class="demo-ad-box" style="width: 728px; height: 90px;">
                        <p>Flexible Ad</p>
                        <span>Any Size</span>
                    </div>
                </div> -->
            </div>

            <!-- Right Ad Space (hidden) -->
            <!-- <div class="ad-space-right">
                <div class="demo-ad-box" style="width: 300px; height: 600px;">
                    <p>Flexible Ad</p>
                    <span>Any Size</span>
                </div>
            </div> -->
        </div>
    `;

    // Make entire upload wrapper clickable
    const uploadWrapper = document.getElementById('merge-upload-wrapper');
    uploadWrapper.onclick = (e) => {
        // Don't trigger if clicking on interactive elements
        if (!e.target.closest('button, .file-thumbnail-preview, .pdf-nav-btn, .drag-handle')) {
            document.getElementById('merge-pdf-input').click();
        }
    };

    document.getElementById('merge-pdf-input').addEventListener('change', handleMergePdfFiles);
    
    // Drag and drop on entire wrapper
    uploadWrapper.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadWrapper.classList.add('dragover');
    });
    
    uploadWrapper.addEventListener('dragleave', () => {
        uploadWrapper.classList.remove('dragover');
    });
    
    uploadWrapper.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadWrapper.classList.remove('dragover');
        const files = e.dataTransfer.files;
        document.getElementById('merge-pdf-input').files = files;
        handleMergePdfFiles({ target: { files: files } });
    });
};

async function handleMergePdfFiles(e) {
    const files = e.target.files;
    if (files.length === 0) {
        alert('Please select PDF files');
        return;
    }

    try {
        await loadPDFLib();

        for (const file of files) {
            try {
                const arrayBuffer = await file.arrayBuffer();
                const pdfDoc = await window.PDFLib.PDFDocument.load(arrayBuffer);
                
                mergePdfState.files.push(file);
                mergePdfState.docs.push({
                    name: file.name,
                    doc: pdfDoc,
                    pages: pdfDoc.getPageCount(),
                    size: (file.size / 1024).toFixed(1)
                });
            } catch (error) {
                alert(`Failed to load "${file.name}": ${error.message}`);
            }
        }

        displayMergePdfList();
    } catch (error) {
        alert('Error loading PDF library: ' + error.message);
    }
}

async function displayMergePdfList() {
    const listDiv = document.getElementById('merge-file-list');
    const uploadHeader = document.getElementById('merge-upload-zone');
    const uploadWrapper = document.getElementById('merge-upload-wrapper');
    const clearBtn = document.getElementById('merge-clear-btn');
    const mergeBtn = document.getElementById('merge-now-btn');
    const processBtn = document.getElementById('merge-process-btn');
    const mobileUploadBtn = document.getElementById('mobile-upload-btn');
    
    if (mergePdfState.docs.length === 0) {
        listDiv.classList.add('hidden');
        uploadHeader.classList.remove('compact');
        uploadWrapper.classList.remove('has-files');
        if (clearBtn) clearBtn.classList.add('hidden');
        if (mergeBtn) mergeBtn.classList.add('hidden');
        if (processBtn) processBtn.classList.add('hidden');
        if (mobileUploadBtn) mobileUploadBtn.classList.remove('hidden');
        return;
    }
    
    listDiv.classList.remove('hidden');
    uploadHeader.classList.add('compact');
    uploadWrapper.classList.add('has-files');
    if (clearBtn) clearBtn.classList.remove('hidden');
    // Don't show download button until merge is complete
    if (mergeBtn) mergeBtn.classList.add('hidden');
    if (processBtn) processBtn.classList.remove('hidden');
    if (mobileUploadBtn) mobileUploadBtn.classList.add('hidden');
    
    let html = `<div class="file-thumbnail-grid">`;
    
    mergePdfState.docs.forEach((pdfInfo, index) => {
        html += `
            <div class="file-thumbnail-item" 
                 data-pdf-index="${index}" 
                 draggable="true"
                 ondragstart="handleDragStart(event, ${index})"
                 ondragover="handleDragOver(event)"
                 ondrop="handleDrop(event, ${index})"
                 ondragend="handleDragEnd(event)"
                 ondragenter="handleDragEnter(event)"
                 ondragleave="handleDragLeave(event)">
                <button onclick="removeMergePdf(${index})" class="file-thumbnail-remove" title="Delete">
                    <i class="fas fa-trash-alt"></i>
                </button>
                <div class="file-thumbnail-preview" id="pdf-preview-${index}" onclick="openPdfPopup(${index})" style="cursor: pointer;">
                    <i class="fas fa-spinner fa-spin file-thumbnail-icon"></i>
                </div>
                <div class="file-thumbnail-name" title="${pdfInfo.name}">${index + 1}. ${pdfInfo.name}</div>
                <div class="file-thumbnail-info">${pdfInfo.size} KB • ${pdfInfo.pages} page${pdfInfo.pages > 1 ? 's' : ''}</div>
                <div class="drag-handle" title="Drag to reorder"><i class="fas fa-grip-vertical"></i></div>
            </div>
        `;
    });

    html += '</div>';
    listDiv.innerHTML = html;
    
    // Render PDF thumbnails instantly
    mergePdfState.files.forEach(async (file, i) => {
        const previewDiv = document.getElementById(`pdf-preview-${i}`);
        if (previewDiv) {
            const result = await renderPdfThumbnail(file, 1);
            if (result) {
                previewDiv.innerHTML = '';
                previewDiv.appendChild(result.canvas);
                previewDiv.dataset.currentPage = '1';
                previewDiv.dataset.totalPages = result.totalPages;
            } else {
                previewDiv.innerHTML = '<i class="fas fa-file-pdf file-thumbnail-icon"></i>';
            }
        }
    });
}

// Navigate PDF pages in thumbnail
window.changePdfPage = async function(index, direction) {
    const previewDiv = document.getElementById(`pdf-preview-${index}`);
    const indicator = document.getElementById(`pdf-page-indicator-${index}`);
    if (!previewDiv) return;
    
    const currentPage = parseInt(previewDiv.dataset.currentPage) || 1;
    const totalPages = parseInt(previewDiv.dataset.totalPages) || 1;
    let newPage = currentPage + direction;
    
    if (newPage < 1) newPage = totalPages;
    if (newPage > totalPages) newPage = 1;
    
    previewDiv.innerHTML = '<i class="fas fa-spinner fa-spin file-thumbnail-icon"></i>';
    
    const file = mergePdfState.files[index];
    const result = await renderPdfThumbnail(file, newPage);
    
    if (result) {
        previewDiv.innerHTML = '';
        previewDiv.appendChild(result.canvas);
        previewDiv.dataset.currentPage = newPage;
        if (indicator) {
            indicator.textContent = `${newPage}/${totalPages}`;
        }
    }
}

// Open PDF in pop-up viewer
window.openPdfPopup = async function(index) {
    const file = mergePdfState.files[index];
    const pdfInfo = mergePdfState.docs[index];
    if (!file) return;
    
    // Create modal overlay
    const modal = document.createElement('div');
    modal.className = 'pdf-popup-overlay';
    modal.innerHTML = `
        <div class="pdf-popup-container">
            <div class="pdf-popup-header">
                <h3><i class="fas fa-file-pdf"></i> ${pdfInfo.name}</h3>
                <button onclick="closePdfPopup()" class="pdf-popup-close" title="Close">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="pdf-popup-viewer" id="popup-viewer">
                <i class="fas fa-spinner fa-spin" style="font-size: 3rem; color: var(--primary-color);"></i>
            </div>
            <div class="pdf-popup-controls">
                <button onclick="changePopupPage(-1)" class="pdf-popup-nav" id="popup-prev">
                    <i class="fas fa-chevron-left"></i> Previous
                </button>
                <span id="popup-page-indicator" class="pdf-popup-indicator">1 / ${pdfInfo.pages}</span>
                <button onclick="changePopupPage(1)" class="pdf-popup-nav" id="popup-next">
                    Next <i class="fas fa-chevron-right"></i>
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    modal.dataset.pdfIndex = index;
    modal.dataset.currentPage = '1';
    modal.dataset.totalPages = pdfInfo.pages;
    
    // Render first page
    const viewer = document.getElementById('popup-viewer');
    const result = await renderPdfThumbnail(file, 1, true);
    if (result) {
        viewer.innerHTML = '';
        viewer.appendChild(result.canvas);
    }
    
    // Close on overlay click
    modal.onclick = (e) => {
        if (e.target === modal) closePdfPopup();
    };
}

// Close pop-up viewer
window.closePdfPopup = function() {
    const modal = document.querySelector('.pdf-popup-overlay');
    if (modal) modal.remove();
}

// Navigate pages in pop-up
window.changePopupPage = async function(direction) {
    const modal = document.querySelector('.pdf-popup-overlay');
    if (!modal) return;
    
    const index = parseInt(modal.dataset.pdfIndex);
    const currentPage = parseInt(modal.dataset.currentPage) || 1;
    const totalPages = parseInt(modal.dataset.totalPages) || 1;
    let newPage = currentPage + direction;
    
    if (newPage < 1) newPage = totalPages;
    if (newPage > totalPages) newPage = 1;
    
    const viewer = document.getElementById('popup-viewer');
    const indicator = document.getElementById('popup-page-indicator');
    
    viewer.innerHTML = '<i class="fas fa-spinner fa-spin" style="font-size: 3rem; color: var(--primary-color);"></i>';
    
    const file = mergePdfState.files[index];
    const result = await renderPdfThumbnail(file, newPage, true);
    
    if (result) {
        viewer.innerHTML = '';
        viewer.appendChild(result.canvas);
        modal.dataset.currentPage = newPage;
        if (indicator) {
            indicator.textContent = `${newPage} / ${totalPages}`;
        }
    }
}

// Drag and drop functionality
let draggedIndex = null;

window.handleDragStart = function(e, index) {
    draggedIndex = index;
    e.currentTarget.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/html', e.currentTarget.innerHTML);
}

window.handleDragOver = function(e) {
    if (e.preventDefault) {
        e.preventDefault();
    }
    e.dataTransfer.dropEffect = 'move';
    return false;
}

window.handleDragEnter = function(e) {
    e.currentTarget.classList.add('drag-over');
}

window.handleDragLeave = function(e) {
    e.currentTarget.classList.remove('drag-over');
}

window.handleDrop = function(e, dropIndex) {
    if (e.stopPropagation) {
        e.stopPropagation();
    }
    e.preventDefault();
    
    e.currentTarget.classList.remove('drag-over');
    
    if (draggedIndex !== null && draggedIndex !== dropIndex) {
        // Reorder arrays
        const draggedFile = mergePdfState.files[draggedIndex];
        const draggedDoc = mergePdfState.docs[draggedIndex];
        
        mergePdfState.files.splice(draggedIndex, 1);
        mergePdfState.docs.splice(draggedIndex, 1);
        
        mergePdfState.files.splice(dropIndex, 0, draggedFile);
        mergePdfState.docs.splice(dropIndex, 0, draggedDoc);
        
        // Re-render list
        displayMergePdfList();
    }
    
    return false;
}

window.handleDragEnd = function(e) {
    e.currentTarget.classList.remove('dragging');
    draggedIndex = null;
    
    // Clean up any remaining drag-over classes
    document.querySelectorAll('.file-thumbnail-item').forEach(item => {
        item.classList.remove('drag-over');
    });
}

function removeMergePdf(index) {
    mergePdfState.docs.splice(index, 1);
    mergePdfState.files.splice(index, 1);
    displayMergePdfList();
}

function clearMergePdfs() {
    mergePdfState.files = [];
    mergePdfState.docs = [];
    displayMergePdfList();
    document.getElementById('merge-pdf-input').value = '';
}

function showResultPage() {
    const container = document.getElementById('tool-container');
    if (!container) {
        console.error('tool-container element not found');
        return;
    }
    
    const resultData = JSON.parse(sessionStorage.getItem('mergeResult'));
    if (!resultData) {
        console.error('No merge result data found');
        return;
    }
    
    container.innerHTML = `
        <div class="tool-page-layout">
            <!-- Left Ad Space (hidden) -->
            <!-- <div class="ad-space-left">
                <div class="demo-ad-box" style="width: 300px; height: 600px;">
                    <p>Flexible Ad</p>
                    <span>Any Size</span>
                </div>
            </div> -->

            <!-- Main Content -->
            <div class="pdf-tool-container">
                <div class="result-page">
                    <div class="result-success-card">
                        <button onclick="window.initializeMergePdf()" class="result-back-btn" title="Go Back">
                            <i class="fas fa-arrow-left"></i>
                        </button>
                        <div class="result-icon">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <h2>✨ PDFs Merged Successfully!</h2>
                        <div class="result-info">
                            <p><strong>${resultData.fileCount}</strong> files combined</p>
                            <p><strong>${resultData.totalPages}</strong> total pages</p>
                            <p><strong>${resultData.fileSize}</strong> MB</p>
                        </div>
                        <div class="result-actions">
                            <button onclick="downloadMergedFile()" class="btn-download-large">
                                <i class="fas fa-download"></i>
                                <span>Download Merged PDF</span>
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Bottom Ad Space (hidden) -->
                <!-- <div class="ad-space-bottom">
                    <div class="demo-ad-box" style="width: 728px; height: 90px;">
                        <p>Flexible Ad</p>
                        <span>Any Size</span>
                    </div>
                </div> -->
            </div>

            <!-- Right Ad Space (hidden) -->
            <!-- <div class="ad-space-right">
                <div class="demo-ad-box" style="width: 300px; height: 600px;">
                    <p>Flexible Ad</p>
                    <span>Any Size</span>
                </div>
            </div> -->
        </div>
    `;
}

function downloadMergedFile() {
    const blobData = sessionStorage.getItem('mergeResultBlob');
    const resultData = JSON.parse(sessionStorage.getItem('mergeResult'));
    
    if (blobData) {
        const link = document.createElement('a');
        link.href = blobData;
        link.download = resultData.fileName;
        link.click();
    }
}

function showLoader(message = 'Processing...') {
    const loader = document.createElement('div');
    loader.id = 'merge-loader-overlay';
    loader.innerHTML = `
        <div class="loader-content">
            <div class="loader-spinner"></div>
            <h3>${message}</h3>
        </div>
    `;
    document.body.appendChild(loader);
}

function hideLoader() {
    const loader = document.getElementById('merge-loader-overlay');
    if (loader) {
        loader.remove();
    }
}

async function mergePdfsNow() {
    if (mergePdfState.docs.length < 2) {
        alert('Please select at least 2 PDF files to merge');
        return;
    }

    // Show loader
    showLoader('Merging PDFs...');

    try {
        await loadPDFLib();

        const mergedPdf = await window.PDFLib.PDFDocument.create();
        
        let totalPages = 0;
        for (const pdfInfo of mergePdfState.docs) {
            const pdfDoc = pdfInfo.doc;
            const pageIndices = pdfDoc.getPageIndices();
            const copiedPages = await mergedPdf.copyPages(pdfDoc, pageIndices);
            
            copiedPages.forEach(page => {
                mergedPdf.addPage(page);
                totalPages++;
            });
        }
        
        const pdfBytes = await mergedPdf.save();
        const blob = new Blob([pdfBytes], { type: 'application/pdf' });
        const fileSize = (blob.size / 1024 / 1024).toFixed(2);

        // Store result data and navigate to result page
        sessionStorage.setItem('mergeResult', JSON.stringify({
            fileName: 'merged.pdf',
            fileSize: fileSize,
            totalPages: totalPages,
            fileCount: mergePdfState.docs.length,
            timestamp: Date.now()
        }));
        
        // Store blob as base64 for download
        const reader = new FileReader();
        reader.onloadend = function() {
            sessionStorage.setItem('mergeResultBlob', reader.result);
            // Hide loader and navigate to result page
            hideLoader();
            showResultPage();
        };
        reader.readAsDataURL(blob);
    } catch (error) {
        console.error('Merge error:', error);
        hideLoader();
        alert('Error merging PDFs: ' + error.message);
    }
}
