// Meme Generator Tool
function initializeMemeGenerator() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="meme-generator-container">
            <h2>😂 Meme Generator</h2>
            <p class="tool-description">Create custom memes with top and bottom text</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="meme-file-input" accept="image/*" onchange="loadMemeImage(this)">
                    <span>📁 Choose Meme Image</span>
                </label>
            </div>

            <div id="meme-editor" class="meme-editor hidden">
                <div class="meme-preview-section">
                    <div class="meme-canvas-container">
                        <img id="meme-preview" src="" alt="Meme Preview">
                        <div id="meme-text-overlay">
                            <div class="meme-text-top" id="preview-top-text"></div>
                            <div class="meme-text-bottom" id="preview-bottom-text"></div>
                        </div>
                    </div>
                </div>

                <div class="meme-controls">
                    <h3>Meme Text</h3>
                    
                    <div class="form-group">
                        <label>Top Text:</label>
                        <input type="text" id="meme-top-text" placeholder="TOP TEXT" 
                               maxlength="50" oninput="updateMemePreview()">
                    </div>

                    <div class="form-group">
                        <label>Bottom Text:</label>
                        <input type="text" id="meme-bottom-text" placeholder="BOTTOM TEXT" 
                               maxlength="50" oninput="updateMemePreview()">
                    </div>

                    <h3>Text Style</h3>
                    
                    <div class="form-group">
                        <label>Font Size: <span id="fontsize-value">48</span>px</label>
                        <input type="range" id="meme-fontsize" min="24" max="96" value="48" 
                               oninput="updateFontSizeValue()">
                    </div>

                    <div class="form-group">
                        <label>Text Color:</label>
                        <input type="color" id="meme-text-color" value="#ffffff" 
                               onchange="updateMemePreview()">
                    </div>

                    <div class="form-group">
                        <label>Stroke Color:</label>
                        <input type="color" id="meme-stroke-color" value="#000000" 
                               onchange="updateMemePreview()">
                    </div>

                    <div class="form-group">
                        <label>Stroke Width: <span id="stroke-value">3</span>px</label>
                        <input type="range" id="meme-stroke-width" min="0" max="10" value="3" 
                               oninput="updateStrokeValue()">
                    </div>

                    <div class="form-group">
                        <label>Font Style:</label>
                        <select id="meme-font" onchange="updateMemePreview()">
                            <option value="Impact">Impact (Classic)</option>
                            <option value="Arial Black">Arial Black</option>
                            <option value="Comic Sans MS">Comic Sans</option>
                            <option value="Verdana">Verdana</option>
                        </select>
                    </div>

                    <button class="btn btn-primary" onclick="generateMeme()">Generate Meme</button>
                </div>
            </div>

            <div id="meme-result" class="result-section hidden">
                <h3>Your Meme is Ready!</h3>
                <img id="meme-final" src="" alt="Generated Meme" style="max-width: 100%; border-radius: 8px;">
                <div class="button-group">
                    <button class="btn btn-secondary" onclick="downloadMeme()">💾 Download Meme</button>
                    <button class="btn btn-secondary" onclick="resetMeme()">🔄 Create Another</button>
                </div>
            </div>
        </div>
    `;
}

let memeImageFile = null;
let memeResultUrl = '';

function loadMemeImage(input) {
    memeImageFile = input.files[0];
    if (!memeImageFile) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        document.getElementById('meme-preview').src = e.target.result;
        document.getElementById('meme-editor').classList.remove('hidden');
        updateMemePreview();
    };
    reader.readAsDataURL(memeImageFile);
}

function updateMemePreview() {
    const topText = document.getElementById('meme-top-text').value.toUpperCase();
    const bottomText = document.getElementById('meme-bottom-text').value.toUpperCase();
    const fontSize = document.getElementById('meme-fontsize').value;
    const textColor = document.getElementById('meme-text-color').value;
    const strokeColor = document.getElementById('meme-stroke-color').value;
    const strokeWidth = document.getElementById('meme-stroke-width').value;
    const font = document.getElementById('meme-font').value;

    const topTextElement = document.getElementById('preview-top-text');
    const bottomTextElement = document.getElementById('preview-bottom-text');

    const style = `
        font-family: '${font}', Impact, sans-serif;
        font-size: ${fontSize}px;
        color: ${textColor};
        text-shadow: 
            -${strokeWidth}px -${strokeWidth}px 0 ${strokeColor},
            ${strokeWidth}px -${strokeWidth}px 0 ${strokeColor},
            -${strokeWidth}px ${strokeWidth}px 0 ${strokeColor},
            ${strokeWidth}px ${strokeWidth}px 0 ${strokeColor};
    `;

    topTextElement.style.cssText = style;
    bottomTextElement.style.cssText = style;
    
    topTextElement.textContent = topText;
    bottomTextElement.textContent = bottomText;
}

function updateFontSizeValue() {
    const value = document.getElementById('meme-fontsize').value;
    document.getElementById('fontsize-value').textContent = value;
    updateMemePreview();
}

function updateStrokeValue() {
    const value = document.getElementById('meme-stroke-width').value;
    document.getElementById('stroke-value').textContent = value;
    updateMemePreview();
}

async function generateMeme() {
    if (!memeImageFile) {
        alert('Please select an image first');
        return;
    }

    const topText = document.getElementById('meme-top-text').value;
    const bottomText = document.getElementById('meme-bottom-text').value;

    if (!topText && !bottomText) {
        alert('Please enter at least one text (top or bottom)');
        return;
    }

    const formData = new FormData();
    formData.append('image', memeImageFile);
    formData.append('topText', topText);
    formData.append('bottomText', bottomText);
    formData.append('fontSize', document.getElementById('meme-fontsize').value);
    formData.append('textColor', document.getElementById('meme-text-color').value);
    formData.append('strokeColor', document.getElementById('meme-stroke-color').value);
    formData.append('strokeWidth', document.getElementById('meme-stroke-width').value);
    formData.append('font', document.getElementById('meme-font').value);

    try {
        const response = await fetch('/api/creative/meme-generate', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            memeResultUrl = data.url;
            document.getElementById('meme-final').src = data.url;
            document.getElementById('meme-result').classList.remove('hidden');
            
            // Scroll to result
            document.getElementById('meme-result').scrollIntoView({ behavior: 'smooth' });
        } else {
            alert('Failed to generate meme: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while generating the meme');
    }
}

function downloadMeme() {
    if (!memeResultUrl) return;
    
    const link = document.createElement('a');
    link.href = memeResultUrl;
    link.download = 'meme_' + Date.now() + '.jpg';
    link.click();
}

function resetMeme() {
    memeImageFile = null;
    memeResultUrl = '';
    document.getElementById('meme-file-input').value = '';
    document.getElementById('meme-top-text').value = '';
    document.getElementById('meme-bottom-text').value = '';
    document.getElementById('meme-editor').classList.add('hidden');
    document.getElementById('meme-result').classList.add('hidden');
}


// Expose to window for router
window.initializeMemeGenerator = initializeMemeGenerator;
