// JSON to CSV Converter

window.initializeJsonToCsv = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-exchange-alt"></i> JSON to CSV Converter</h2>
                <p>Convert JSON data to CSV format</p>
            </div>

            <div class="conversion-tabs">
                <button class="tab-btn active" onclick="switchConversionTab('tocsv')">JSON → CSV</button>
                <button class="tab-btn" onclick="switchConversionTab('tojson')">CSV → JSON</button>
            </div>

            <div id="tocsv-tab" class="tab-content">
                <div class="form-group">
                    <label>JSON Input:</label>
                    <textarea id="json-input" rows="12" placeholder='[
  {"name": "John", "age": 30, "city": "New York"},
  {"name": "Jane", "age": 25, "city": "London"}
]'></textarea>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="json-include-headers" checked> Include headers
                    </label>
                </div>

                <div class="form-group">
                    <label>Delimiter:</label>
                    <select id="json-delimiter">
                        <option value=",">Comma (,)</option>
                        <option value=";">Semicolon (;)</option>
                        <option value="\t">Tab</option>
                        <option value="|">Pipe (|)</option>
                    </select>
                </div>

                <button onclick="convertJsonToCsv()" class="btn-primary">
                    <i class="fas fa-arrow-right"></i> Convert to CSV
                </button>

                <div id="csv-output-section" class="results-section hidden">
                    <label>CSV Output:</label>
                    <textarea id="csv-output" rows="12" readonly></textarea>
                    <div class="button-group">
                        <button onclick="copyCsv()" class="btn-secondary">
                            <i class="fas fa-copy"></i> Copy
                        </button>
                        <button onclick="downloadCsv()" class="btn-primary">
                            <i class="fas fa-download"></i> Download CSV
                        </button>
                    </div>
                </div>
            </div>

            <div id="tojson-tab" class="tab-content hidden">
                <div class="form-group">
                    <label>CSV Input:</label>
                    <textarea id="csv-input-reverse" rows="12" placeholder='name,age,city
John,30,New York
Jane,25,London'></textarea>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="csv-has-headers" checked> First row contains headers
                    </label>
                </div>

                <button onclick="convertCsvToJsonReverse()" class="btn-primary">
                    <i class="fas fa-arrow-right"></i> Convert to JSON
                </button>

                <div id="json-output-section" class="results-section hidden">
                    <label>JSON Output:</label>
                    <textarea id="json-output-reverse" rows="12" readonly></textarea>
                    <div class="button-group">
                        <button onclick="copyJsonReverse()" class="btn-secondary">
                            <i class="fas fa-copy"></i> Copy
                        </button>
                        <button onclick="downloadJsonReverse()" class="btn-primary">
                            <i class="fas fa-download"></i> Download JSON
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
};

function switchConversionTab(tab) {
    document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    document.getElementById('tocsv-tab').classList.toggle('hidden', tab !== 'tocsv');
    document.getElementById('tojson-tab').classList.toggle('hidden', tab !== 'tojson');
}

function convertJsonToCsv() {
    const jsonInput = document.getElementById('json-input').value.trim();
    if (!jsonInput) {
        alert('Please enter JSON data');
        return;
    }

    try {
        const data = JSON.parse(jsonInput);
        if (!Array.isArray(data)) {
            alert('JSON must be an array of objects');
            return;
        }

        const includeHeaders = document.getElementById('json-include-headers').checked;
        const delimiter = document.getElementById('json-delimiter').value;

        const headers = Object.keys(data[0]);
        let csv = '';

        if (includeHeaders) {
            csv = headers.join(delimiter) + '\n';
        }

        data.forEach(row => {
            const values = headers.map(header => {
                const value = row[header] !== undefined ? row[header] : '';
                return typeof value === 'string' && value.includes(delimiter) ? `"${value}"` : value;
            });
            csv += values.join(delimiter) + '\n';
        });

        document.getElementById('csv-output').value = csv;
        document.getElementById('csv-output-section').classList.remove('hidden');
    } catch (error) {
        alert('Invalid JSON: ' + error.message);
    }
}

function convertCsvToJsonReverse() {
    const csvInput = document.getElementById('csv-input-reverse').value.trim();
    if (!csvInput) {
        alert('Please enter CSV data');
        return;
    }

    const hasHeaders = document.getElementById('csv-has-headers').checked;
    const lines = csvInput.split('\n').filter(line => line.trim());

    const headers = hasHeaders ? lines[0].split(',').map(h => h.trim()) : [];
    const dataLines = hasHeaders ? lines.slice(1) : lines;

    const result = dataLines.map((line, index) => {
        const values = line.split(',').map(v => v.trim());
        if (hasHeaders) {
            const obj = {};
            headers.forEach((header, i) => {
                obj[header] = isNaN(values[i]) ? values[i] : Number(values[i]);
            });
            return obj;
        } else {
            return values.map(v => isNaN(v) ? v : Number(v));
        }
    });

    document.getElementById('json-output-reverse').value = JSON.stringify(result, null, 2);
    document.getElementById('json-output-section').classList.remove('hidden');
}

function copyCsv() {
    const output = document.getElementById('csv-output');
    output.select();
    document.execCommand('copy');
}

function downloadCsv() {
    const csv = document.getElementById('csv-output').value;
    const blob = new Blob([csv], { type: 'text/csv' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'output.csv';
    link.click();
}

function copyJsonReverse() {
    const output = document.getElementById('json-output-reverse');
    output.select();
    document.execCommand('copy');
}

function downloadJsonReverse() {
    const json = document.getElementById('json-output-reverse').value;
    const blob = new Blob([json], { type: 'application/json' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'output.json';
    link.click();
}
