// Image Watermark Tool

let watermarkImages = [];
let logoFile = null;

window.initializeImageWatermark = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-copyright"></i> Image Watermark</h2>
                <p>Add text or logo watermarks to images</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('watermark-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop images here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP</p>
                <input type="file" id="watermark-input" accept="image/*" multiple style="display: none;">
            </div>

            <div id="watermark-preview" class="preview-section hidden">
                <h3>Selected Images (<span id="watermark-count">0</span>)</h3>
                <div id="watermark-grid" class="image-grid"></div>
            </div>

            <div id="watermark-options" class="options-section hidden">
                <h3>Watermark Options</h3>
                
                <div class="form-group">
                    <label>Watermark Type:</label>
                    <select id="watermark-type" onchange="toggleWatermarkType()">
                        <option value="text">Text</option>
                        <option value="logo">Logo/Image</option>
                    </select>
                </div>

                <div id="text-watermark-options">
                    <div class="form-group">
                        <label>Text:</label>
                        <input type="text" id="watermark-text" class="form-input" value="© Copyright 2024" placeholder="Enter watermark text">
                    </div>
                    <div class="form-group">
                        <label>Font Size: <span id="font-size-value">24</span>px</label>
                        <input type="range" id="watermark-font-size" min="12" max="100" value="24" oninput="document.getElementById('font-size-value').textContent = this.value">
                    </div>
                    <div class="form-group">
                        <label>Color:</label>
                        <input type="color" id="watermark-color" value="#ffffff">
                    </div>
                </div>

                <div id="logo-watermark-options" class="hidden">
                    <div class="form-group">
                        <label>Upload Logo:</label>
                        <input type="file" id="logo-input" accept="image/*" onchange="handleLogoUpload(event)" class="form-input">
                    </div>
                    <div class="form-group">
                        <label>Logo Size: <span id="logo-size-value">100</span>px</label>
                        <input type="range" id="logo-size" min="50" max="500" value="100" oninput="document.getElementById('logo-size-value').textContent = this.value">
                    </div>
                </div>

                <div class="form-group">
                    <label>Position:</label>
                    <select id="watermark-position">
                        <option value="bottom-right">Bottom Right</option>
                        <option value="bottom-left">Bottom Left</option>
                        <option value="top-right">Top Right</option>
                        <option value="top-left">Top Left</option>
                        <option value="center">Center</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Opacity: <span id="opacity-value">50</span>%</label>
                    <input type="range" id="watermark-opacity" min="0" max="100" value="50" oninput="document.getElementById('opacity-value').textContent = this.value">
                </div>

                <button onclick="applyWatermark()" class="btn-primary">
                    <i class="fas fa-stamp"></i> Apply Watermark
                </button>
            </div>

            <div id="watermark-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('watermark-input').addEventListener('change', handleWatermarkUpload);
};

function handleWatermarkUpload(e) {
    watermarkImages = Array.from(e.target.files);
    if (!watermarkImages.length) return;

    document.getElementById('watermark-count').textContent = watermarkImages.length;
    document.getElementById('watermark-preview').classList.remove('hidden');
    document.getElementById('watermark-options').classList.remove('hidden');

    const grid = document.getElementById('watermark-grid');
    grid.innerHTML = '';

    watermarkImages.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <img src="${URL.createObjectURL(file)}" alt="${file.name}">
            <div class="image-name">${file.name}</div>
        `;
        grid.appendChild(item);
    });
}

function toggleWatermarkType() {
    const type = document.getElementById('watermark-type').value;
    document.getElementById('text-watermark-options').classList.toggle('hidden', type !== 'text');
    document.getElementById('logo-watermark-options').classList.toggle('hidden', type !== 'logo');
}

function handleLogoUpload(e) {
    logoFile = e.target.files[0];
}

async function applyWatermark() {
    if (!watermarkImages.length) return;

    const type = document.getElementById('watermark-type').value;

    if (type === 'logo' && !logoFile) {
        alert('Please upload a logo image');
        return;
    }

    const resultsDiv = document.getElementById('watermark-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Adding watermarks...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    watermarkImages.forEach(file => formData.append('images', file));
    formData.append('type', type);
    formData.append('position', document.getElementById('watermark-position').value);
    formData.append('opacity', document.getElementById('watermark-opacity').value);

    if (type === 'text') {
        formData.append('text', document.getElementById('watermark-text').value);
        formData.append('fontSize', document.getElementById('watermark-font-size').value);
        formData.append('color', document.getElementById('watermark-color').value);
    } else {
        formData.append('logo', logoFile);
        formData.append('logoSize', document.getElementById('logo-size').value);
    }

    try {
        const response = await fetch('/api/image/add-watermark', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayWatermarkResults(data.images);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displayWatermarkResults(images) {
    const resultsDiv = document.getElementById('watermark-results');
    let html = '<div class="results-success"><h3>✅ Watermarks Added!</h3><div class="results-grid">';

    images.forEach(img => {
        html += `
            <div class="result-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="result-info">
                    <div class="result-name">${img.filename}</div>
                    <a href="${img.url}" download="${img.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download
                    </a>
                </div>
            </div>
        `;
    });

    html += '</div></div>';
    resultsDiv.innerHTML = html;
}
