// Image to Text (OCR)

let ocrImage = null;

window.initializeImageToText = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-file-image"></i> Image to Text (OCR)</h2>
                <p>Extract text from images using OCR</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('ocr-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop image here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP, PDF</p>
                <input type="file" id="ocr-input" accept="image/*,.pdf" style="display: none;">
            </div>

            <div id="ocr-preview" class="preview-section hidden">
                <h3>Image Preview</h3>
                <img id="ocr-image-preview" style="max-width: 100%;">
            </div>

            <div id="ocr-options" class="options-section hidden">
                <h3>OCR Options</h3>
                
                <div class="form-group">
                    <label>Language:</label>
                    <select id="ocr-language">
                        <option value="eng">English</option>
                        <option value="spa">Spanish</option>
                        <option value="fra">French</option>
                        <option value="deu">German</option>
                        <option value="chi_sim">Chinese (Simplified)</option>
                        <option value="jpn">Japanese</option>
                        <option value="rus">Russian</option>
                        <option value="ara">Arabic</option>
                    </select>
                </div>

                <button onclick="extractText()" class="btn-primary">
                    <i class="fas fa-file-alt"></i> Extract Text
                </button>
            </div>

            <div id="ocr-results" class="results-section hidden">
                <label>Extracted Text:</label>
                <textarea id="ocr-text" rows="15" readonly></textarea>
                <div class="button-group">
                    <button onclick="copyOcrText()" class="btn-secondary">
                        <i class="fas fa-copy"></i> Copy
                    </button>
                    <button onclick="downloadOcrText()" class="btn-primary">
                        <i class="fas fa-download"></i> Download TXT
                    </button>
                </div>
            </div>
        </div>
    `;

    document.getElementById('ocr-input').addEventListener('change', handleOcrUpload);
};

function handleOcrUpload(e) {
    ocrImage = e.target.files[0];
    if (!ocrImage) return;

    const preview = document.getElementById('ocr-image-preview');
    preview.src = URL.createObjectURL(ocrImage);

    document.getElementById('ocr-preview').classList.remove('hidden');
    document.getElementById('ocr-options').classList.remove('hidden');
}

async function extractText() {
    if (!ocrImage) return;

    const resultsDiv = document.getElementById('ocr-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Extracting text... This may take a moment.</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('image', ocrImage);
    formData.append('language', document.getElementById('ocr-language').value);

    try {
        const response = await fetch('/api/ocr/extract-text', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            document.getElementById('ocr-text').value = data.text;
            resultsDiv.innerHTML = `
                <label>Extracted Text:</label>
                <textarea id="ocr-text" rows="15" readonly>${data.text}</textarea>
                <div class="button-group">
                    <button onclick="copyOcrText()" class="btn-secondary">
                        <i class="fas fa-copy"></i> Copy
                    </button>
                    <button onclick="downloadOcrText()" class="btn-primary">
                        <i class="fas fa-download"></i> Download TXT
                    </button>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function copyOcrText() {
    const text = document.getElementById('ocr-text');
    text.select();
    document.execCommand('copy');
    alert('Text copied to clipboard!');
}

function downloadOcrText() {
    const text = document.getElementById('ocr-text').value;
    const blob = new Blob([text], { type: 'text/plain' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'extracted-text.txt';
    link.click();
}
