// Image to PDF - Converts any image format to PDF

let imageFiles = [];

window.initializeImageToPdf = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-file-pdf"></i> Image to PDF</h2>
                <p>Convert any image format (JPG, PNG, WebP, GIF, BMP, TIFF, HEIC, SVG) to PDF</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('image-to-pdf-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop images here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP, GIF, BMP, TIFF, HEIC, SVG and more</p>
                <input type="file" id="image-to-pdf-input" accept="image/*" multiple style="display: none;">
            </div>

            <div id="image-files-list" class="files-list hidden"></div>

            <div id="image-to-pdf-options" class="options-section hidden">
                <h3>PDF Options</h3>
                
                <div class="form-group">
                    <label>Page Size:</label>
                    <select id="pdf-page-size">
                        <option value="auto">Auto (Fit to Image)</option>
                        <option value="A4">A4 (210 x 297 mm)</option>
                        <option value="A3">A3 (297 x 420 mm)</option>
                        <option value="Letter">Letter (8.5 x 11 in)</option>
                        <option value="Legal">Legal (8.5 x 14 in)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Orientation:</label>
                    <select id="pdf-orientation">
                        <option value="auto">Auto</option>
                        <option value="portrait">Portrait</option>
                        <option value="landscape">Landscape</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Image Placement:</label>
                    <select id="image-placement">
                        <option value="center">Center</option>
                        <option value="stretch">Stretch to Page</option>
                        <option value="fit">Fit (Maintain Aspect)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="combine-images" checked>
                        Combine multiple images into single PDF
                    </label>
                </div>

                <button onclick="convertImageToPdf()" class="btn-primary">
                    <i class="fas fa-file-pdf"></i> Convert to PDF
                </button>
            </div>

            <div id="image-to-pdf-result" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('image-to-pdf-input').addEventListener('change', handleImageUpload);
};

function handleImageUpload(e) {
    imageFiles = Array.from(e.target.files);
    displayImageFiles();
}

function displayImageFiles() {
    const listDiv = document.getElementById('image-files-list');
    let html = '<h3>Images (' + imageFiles.length + ')</h3><div class="file-items-grid">';
    
    imageFiles.forEach((file, index) => {
        html += `
            <div class="file-item">
                <div class="file-preview">
                    <img src="${URL.createObjectURL(file)}" alt="${file.name}">
                </div>
                <div class="file-info">
                    <p class="file-name">${file.name}</p>
                    <p class="file-size">${(file.size / 1024).toFixed(2)} KB</p>
                </div>
                <button onclick="removeImageFile(${index})" class="btn-remove">Remove</button>
            </div>
        `;
    });

    html += '</div>';
    listDiv.innerHTML = html;
    listDiv.classList.remove('hidden');
    document.getElementById('image-to-pdf-options').classList.remove('hidden');
}

function removeImageFile(index) {
    imageFiles.splice(index, 1);
    if (imageFiles.length === 0) {
        document.getElementById('image-files-list').classList.add('hidden');
        document.getElementById('image-to-pdf-options').classList.add('hidden');
    } else {
        displayImageFiles();
    }
}

async function convertImageToPdf() {
    if (imageFiles.length === 0) {
        alert('Please upload at least one image');
        return;
    }

    const resultDiv = document.getElementById('image-to-pdf-result');
    resultDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Creating PDF...</div>';
    resultDiv.classList.remove('hidden');

    const formData = new FormData();
    imageFiles.forEach(file => {
        formData.append('images', file);
    });
    formData.append('pageSize', document.getElementById('pdf-page-size').value);
    formData.append('orientation', document.getElementById('pdf-orientation').value);
    formData.append('placement', document.getElementById('image-placement').value);
    formData.append('combine', document.getElementById('combine-images').checked);

    try {
        const response = await fetch('/api/convert/image-to-pdf', {
            method: 'POST',
            body: formData
        });

        if (!response.ok) {
            throw new Error('Conversion failed');
        }

        const blob = await response.blob();
        const url = URL.createObjectURL(blob);

        resultDiv.innerHTML = `
            <h3>PDF Created Successfully!</h3>
            <p>File size: ${(blob.size / 1024 / 1024).toFixed(2)} MB</p>
            <a href="${url}" download="images.pdf" class="btn-primary">
                <i class="fas fa-download"></i> Download PDF
            </a>
        `;
    } catch (error) {
        resultDiv.innerHTML = `<div class="error"><i class="fas fa-exclamation-circle"></i> Error: ${error.message}</div>`;
    }
}

// CSS Styles
const styles = `
.file-items-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
    gap: 15px;
    margin-bottom: 20px;
}

.file-item {
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 10px;
    text-align: center;
    transition: box-shadow 0.3s;
}

.file-item:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.file-preview {
    width: 100%;
    height: 120px;
    overflow: hidden;
    border-radius: 6px;
    margin-bottom: 10px;
    background: #f3f4f6;
    display: flex;
    align-items: center;
    justify-content: center;
}

.file-preview img {
    max-width: 100%;
    max-height: 100%;
    object-fit: contain;
}

.file-info {
    margin-bottom: 8px;
}

.file-name {
    font-size: 0.85rem;
    font-weight: 500;
    color: #1f2937;
    margin: 5px 0;
    word-break: break-word;
}

.file-size {
    font-size: 0.75rem;
    color: #9ca3af;
    margin: 3px 0;
}

.btn-remove {
    width: 100%;
    padding: 6px;
    background: #ef4444;
    color: white;
    border: none;
    border-radius: 4px;
    font-size: 0.85rem;
    cursor: pointer;
    transition: background 0.3s;
}

.btn-remove:hover {
    background: #dc2626;
}
`;

// Inject styles
if (!document.getElementById('image-to-pdf-styles')) {
    const styleElement = document.createElement('style');
    styleElement.id = 'image-to-pdf-styles';
    styleElement.textContent = styles;
    document.head.appendChild(styleElement);
}
