// Image Splitter

let splitImage = null;

window.initializeImageSplitter = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-th"></i> Image Splitter</h2>
                <p>Split images into multiple parts</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('split-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop image here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP</p>
                <input type="file" id="split-input" accept="image/*" style="display: none;">
            </div>

            <div id="split-preview" class="preview-section hidden">
                <h3>Image Preview</h3>
                <img id="split-image-preview" style="max-width: 100%;">
            </div>

            <div id="split-options" class="options-section hidden">
                <h3>Split Options</h3>
                <div class="form-group">
                    <label>Split Mode:</label>
                    <select id="split-mode" onchange="updateSplitMode()">
                        <option value="grid">Grid (Rows x Columns)</option>
                        <option value="equal">Equal Parts</option>
                        <option value="custom">Custom Sizes</option>
                    </select>
                </div>

                <div id="grid-options">
                    <div class="form-group">
                        <label>Rows:</label>
                        <input type="number" id="split-rows" min="1" max="10" value="2" class="form-input">
                    </div>
                    <div class="form-group">
                        <label>Columns:</label>
                        <input type="number" id="split-columns" min="1" max="10" value="2" class="form-input">
                    </div>
                </div>

                <div id="equal-options" class="hidden">
                    <div class="form-group">
                        <label>Number of Parts:</label>
                        <input type="number" id="split-parts" min="2" max="20" value="4" class="form-input">
                    </div>
                </div>

                <button onclick="splitImageNow()" class="btn-primary">
                    <i class="fas fa-cut"></i> Split Image
                </button>
            </div>

            <div id="split-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('split-input').addEventListener('change', handleSplitUpload);
};

function handleSplitUpload(e) {
    splitImage = e.target.files[0];
    if (!splitImage) return;

    const preview = document.getElementById('split-image-preview');
    preview.src = URL.createObjectURL(splitImage);

    document.getElementById('split-preview').classList.remove('hidden');
    document.getElementById('split-options').classList.remove('hidden');
}

function updateSplitMode() {
    const mode = document.getElementById('split-mode').value;
    document.getElementById('grid-options').classList.toggle('hidden', mode !== 'grid');
    document.getElementById('equal-options').classList.toggle('hidden', mode !== 'equal');
}

async function splitImageNow() {
    if (!splitImage) return;

    const resultsDiv = document.getElementById('split-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Splitting image...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    formData.append('image', splitImage);
    formData.append('mode', document.getElementById('split-mode').value);

    if (document.getElementById('split-mode').value === 'grid') {
        formData.append('rows', document.getElementById('split-rows').value);
        formData.append('columns', document.getElementById('split-columns').value);
    } else {
        formData.append('parts', document.getElementById('split-parts').value);
    }

    try {
        const response = await fetch('/api/image/split', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displaySplitResults(data.images);
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}

function displaySplitResults(images) {
    const resultsDiv = document.getElementById('split-results');
    let html = '<div class="results-success"><h3>✅ Image Split Successfully!</h3><div class="results-grid">';

    images.forEach(img => {
        html += `
            <div class="result-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="result-info">
                    <div class="result-name">${img.filename}</div>
                    <a href="${img.url}" download="${img.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download
                    </a>
                </div>
            </div>
        `;
    });

    html += '</div><button onclick="downloadAllSplit()" class="btn-primary"><i class="fas fa-download"></i> Download All as ZIP</button></div>';
    resultsDiv.innerHTML = html;
}

async function downloadAllSplit() {
    alert('ZIP download feature coming soon!');
}
