// Bulk Image Resizer Tool
function initializeImageResizer() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="image-resizer-container">
            <h2>📐 Bulk Image Resizer</h2>
            <p class="tool-description">Resize multiple images to exact dimensions or percentages</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="resize-input" accept="image/*" multiple onchange="loadImagesForResize(this)">
                    <span>📁 Choose Images (Multiple)</span>
                </label>
            </div>

            <div id="resize-options" class="resize-options hidden">
                <div class="image-count-info">
                    <strong><span id="resize-image-count">0</span> images selected</strong>
                </div>

                <div class="form-group">
                    <label>Resize Mode:</label>
                    <select id="resize-mode" onchange="updateResizeMode()">
                        <option value="exact">Exact Dimensions</option>
                        <option value="percentage">Percentage</option>
                        <option value="fit">Fit Within Dimensions</option>
                        <option value="cover">Cover Dimensions</option>
                        <option value="width">Fixed Width (Auto Height)</option>
                        <option value="height">Fixed Height (Auto Width)</option>
                    </select>
                </div>

                <div id="resize-mode-options">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Width (px):</label>
                            <input type="number" id="resize-width" value="800" min="1">
                        </div>
                        <div class="form-group">
                            <label>Height (px):</label>
                            <input type="number" id="resize-height" value="600" min="1">
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="resize-format">
                        <option value="original">Keep Original</option>
                        <option value="jpg">JPG</option>
                        <option value="png">PNG</option>
                        <option value="webp">WebP</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Quality: <span id="resize-quality-value">85</span>%</label>
                    <input type="range" id="resize-quality" min="1" max="100" value="85" oninput="updateQualityValue()">
                </div>

                <div class="form-group">
                    <label>
                        <input type="checkbox" id="maintain-aspect" checked>
                        Maintain Aspect Ratio
                    </label>
                </div>

                <button class="btn btn-primary" onclick="resizeImages()">Resize Images</button>
            </div>

            <div id="resize-results" class="result-section hidden">
                <h3>Resized Images</h3>
                <div id="resize-preview-grid" class="results-grid"></div>
                <button class="btn btn-secondary" onclick="downloadAllResized()">📦 Download All as ZIP</button>
            </div>
        </div>
    `;
}

let resizeFiles = [];

function loadImagesForResize(input) {
    resizeFiles = Array.from(input.files);
    document.getElementById('resize-image-count').textContent = resizeFiles.length;
    document.getElementById('resize-options').classList.remove('hidden');
}

function updateResizeMode() {
    const mode = document.getElementById('resize-mode').value;
    const optionsDiv = document.getElementById('resize-mode-options');
    
    switch(mode) {
        case 'percentage':
            optionsDiv.innerHTML = `
                <div class="form-group">
                    <label>Scale: <span id="scale-value">50</span>%</label>
                    <input type="range" id="resize-scale" min="1" max="200" value="50" oninput="updateScaleValue()">
                </div>`;
            break;
        case 'width':
            optionsDiv.innerHTML = `
                <div class="form-group">
                    <label>Width (px):</label>
                    <input type="number" id="resize-width" value="800" min="1">
                </div>`;
            break;
        case 'height':
            optionsDiv.innerHTML = `
                <div class="form-group">
                    <label>Height (px):</label>
                    <input type="number" id="resize-height" value="600" min="1">
                </div>`;
            break;
        default:
            optionsDiv.innerHTML = `
                <div class="form-row">
                    <div class="form-group">
                        <label>Width (px):</label>
                        <input type="number" id="resize-width" value="800" min="1">
                    </div>
                    <div class="form-group">
                        <label>Height (px):</label>
                        <input type="number" id="resize-height" value="600" min="1">
                    </div>
                </div>`;
    }
}

function updateQualityValue() {
    document.getElementById('resize-quality-value').textContent = 
        document.getElementById('resize-quality').value;
}

function updateScaleValue() {
    document.getElementById('scale-value').textContent = 
        document.getElementById('resize-scale').value;
}

async function resizeImages() {
    if (resizeFiles.length === 0) return;

    const formData = new FormData();
    resizeFiles.forEach(file => formData.append('images', file));
    
    const mode = document.getElementById('resize-mode').value;
    formData.append('mode', mode);
    formData.append('format', document.getElementById('resize-format').value);
    formData.append('quality', document.getElementById('resize-quality').value);
    formData.append('maintainAspect', document.getElementById('maintain-aspect').checked);

    if (mode === 'percentage') {
        formData.append('scale', document.getElementById('resize-scale').value);
    } else if (mode === 'width' || mode === 'height' || mode === 'exact' || mode === 'fit' || mode === 'cover') {
        const width = document.getElementById('resize-width')?.value;
        const height = document.getElementById('resize-height')?.value;
        if (width) formData.append('width', width);
        if (height) formData.append('height', height);
    }

    try {
        const response = await fetch('/api/image/bulk-resize', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayResizeResults(data.images);
        } else {
            alert('Failed to resize images: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while resizing images');
    }
}

function displayResizeResults(images) {
    const grid = document.getElementById('resize-preview-grid');
    grid.innerHTML = '';

    images.forEach(img => {
        const item = document.createElement('div');
        item.className = 'result-item';
        item.innerHTML = `
            <img src="${img.url}" alt="${img.filename}">
            <div class="result-info">
                <p>${img.filename}</p>
                <p>${img.dimensions}</p>
                <button class="btn-small" onclick="downloadFile('${img.url}', '${img.filename}')">Download</button>
            </div>
        `;
        grid.appendChild(item);
    });

    document.getElementById('resize-results').classList.remove('hidden');
}

function downloadAllResized() {
    alert('Downloading all resized images...');
}


// Expose to window for router
window.initializeImageResizer = initializeImageResizer;
