// Image Optimizer Tool
function initializeImageOptimizer() {
    const container = document.getElementById('tool-container');
    
    container.innerHTML = `
        <div class="image-optimizer-container">
            <h2>⚡ Image Optimizer</h2>
            <p class="tool-description">Optimize images for web with smart compression</p>

            <div class="upload-section">
                <label class="file-input-label">
                    <input type="file" id="optimize-input" accept="image/*" multiple onchange="loadOptimizeImages(this)">
                    <span>📁 Choose Images to Optimize (Multiple)</span>
                </label>
            </div>

            <div id="optimize-options" class="optimize-options hidden">
                <div class="image-count-info">
                    <strong><span id="optimize-count">0</span> images selected</strong>
                </div>

                <div class="optimization-presets">
                    <h3>Optimization Preset</h3>
                    <div class="preset-buttons">
                        <button class="preset-btn" data-preset="web" onclick="selectPreset('web')">
                            <h4>🌐 Web</h4>
                            <p>Balanced quality & size</p>
                            <small>~80% quality</small>
                        </button>
                        <button class="preset-btn" data-preset="social" onclick="selectPreset('social')">
                            <h4>📱 Social Media</h4>
                            <p>Perfect for Instagram, Facebook</p>
                            <small>Max 1080px</small>
                        </button>
                        <button class="preset-btn" data-preset="email" onclick="selectPreset('email')">
                            <h4>📧 Email</h4>
                            <p>Small file size</p>
                            <small>Max 800px</small>
                        </button>
                        <button class="preset-btn active" data-preset="custom" onclick="selectPreset('custom')">
                            <h4>⚙️ Custom</h4>
                            <p>Manual settings</p>
                            <small>You decide</small>
                        </button>
                    </div>
                </div>

                <div id="custom-optimize-settings">
                    <h3>Custom Settings</h3>
                    
                    <div class="form-group">
                        <label>Target Quality: <span id="opt-quality-val">80</span>%</label>
                        <input type="range" id="opt-quality" min="1" max="100" value="80" oninput="updateOptQuality()">
                    </div>

                    <div class="form-group">
                        <label>Max Dimensions:</label>
                        <select id="opt-max-dimension">
                            <option value="0">Original Size</option>
                            <option value="800">800px (Small)</option>
                            <option value="1080">1080px (Medium)</option>
                            <option value="1920" selected>1920px (Large)</option>
                            <option value="2560">2560px (Extra Large)</option>
                            <option value="4096">4096px (4K)</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Output Format:</label>
                        <select id="opt-format">
                            <option value="auto">Auto (Best Format)</option>
                            <option value="jpg">JPG</option>
                            <option value="webp">WebP (Modern)</option>
                            <option value="png">PNG</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="opt-strip-metadata" checked>
                            Remove metadata (EXIF, GPS, etc.)
                        </label>
                    </div>

                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="opt-progressive" checked>
                            Progressive/Interlaced loading
                        </label>
                    </div>
                </div>

                <button class="btn btn-primary" onclick="optimizeImages()">Optimize Images</button>
            </div>

            <div id="optimize-results" class="result-section hidden">
                <h3>Optimization Results</h3>
                
                <div class="optimization-summary">
                    <div class="summary-stat">
                        <h4>Original Size</h4>
                        <p id="original-size">0 MB</p>
                    </div>
                    <div class="summary-stat">
                        <h4>Optimized Size</h4>
                        <p id="optimized-size">0 MB</p>
                    </div>
                    <div class="summary-stat success">
                        <h4>Saved</h4>
                        <p id="saved-size">0 MB (<span id="saved-percent">0</span>%)</p>
                    </div>
                </div>

                <div id="optimize-grid" class="optimize-comparison-grid"></div>
                
                <button class="btn btn-secondary" onclick="downloadAllOptimized()">📦 Download All as ZIP</button>
            </div>
        </div>
    `;
}

let optimizeFiles = [];
let currentPreset = 'custom';

function loadOptimizeImages(input) {
    optimizeFiles = Array.from(input.files);
    document.getElementById('optimize-count').textContent = optimizeFiles.length;
    document.getElementById('optimize-options').classList.remove('hidden');
}

function selectPreset(preset) {
    currentPreset = preset;
    document.querySelectorAll('.preset-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.preset === preset);
    });

    const customSettings = document.getElementById('custom-optimize-settings');
    
    switch(preset) {
        case 'web':
            document.getElementById('opt-quality').value = 80;
            document.getElementById('opt-max-dimension').value = 1920;
            document.getElementById('opt-format').value = 'webp';
            customSettings.style.opacity = '0.5';
            customSettings.style.pointerEvents = 'none';
            break;
        case 'social':
            document.getElementById('opt-quality').value = 85;
            document.getElementById('opt-max-dimension').value = 1080;
            document.getElementById('opt-format').value = 'jpg';
            customSettings.style.opacity = '0.5';
            customSettings.style.pointerEvents = 'none';
            break;
        case 'email':
            document.getElementById('opt-quality').value = 70;
            document.getElementById('opt-max-dimension').value = 800;
            document.getElementById('opt-format').value = 'jpg';
            customSettings.style.opacity = '0.5';
            customSettings.style.pointerEvents = 'none';
            break;
        case 'custom':
            customSettings.style.opacity = '1';
            customSettings.style.pointerEvents = 'auto';
            break;
    }
    
    updateOptQuality();
}

function updateOptQuality() {
    document.getElementById('opt-quality-val').textContent = 
        document.getElementById('opt-quality').value;
}

async function optimizeImages() {
    if (optimizeFiles.length === 0) return;

    const formData = new FormData();
    optimizeFiles.forEach(file => formData.append('images', file));
    formData.append('quality', document.getElementById('opt-quality').value);
    formData.append('maxDimension', document.getElementById('opt-max-dimension').value);
    formData.append('format', document.getElementById('opt-format').value);
    formData.append('stripMetadata', document.getElementById('opt-strip-metadata').checked);
    formData.append('progressive', document.getElementById('opt-progressive').checked);

    try {
        const response = await fetch('/api/image/optimize', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayOptimizeResults(data);
        } else {
            alert('Failed to optimize images: ' + (data.error || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred during optimization');
    }
}

function displayOptimizeResults(data) {
    const originalSize = data.originalSize || 0;
    const optimizedSize = data.optimizedSize || 0;
    const saved = originalSize - optimizedSize;
    const savedPercent = originalSize > 0 ? ((saved / originalSize) * 100).toFixed(1) : 0;

    document.getElementById('original-size').textContent = formatFileSize(originalSize);
    document.getElementById('optimized-size').textContent = formatFileSize(optimizedSize);
    document.getElementById('saved-size').textContent = formatFileSize(saved);
    document.getElementById('saved-percent').textContent = savedPercent;

    const grid = document.getElementById('optimize-grid');
    grid.innerHTML = '';

    data.images.forEach(img => {
        const item = document.createElement('div');
        item.className = 'optimize-item';
        const reduction = ((img.reduction || 0) * 100).toFixed(1);
        
        item.innerHTML = `
            <div class="optimize-before-after">
                <img src="${img.url}" alt="${img.filename}">
            </div>
            <div class="optimize-info">
                <p><strong>${img.filename}</strong></p>
                <p>Before: ${formatFileSize(img.originalSize)} → After: ${formatFileSize(img.optimizedSize)}</p>
                <p class="reduction ${reduction > 20 ? 'success' : ''}">Reduced by ${reduction}%</p>
                <button class="btn-small" onclick="downloadFile('${img.url}', '${img.filename}')">Download</button>
            </div>
        `;
        grid.appendChild(item);
    });

    document.getElementById('optimize-results').classList.remove('hidden');
}

function downloadAllOptimized() {
    alert('Downloading all optimized images...');
}


// Expose to window for router
window.initializeImageOptimizer = initializeImageOptimizer;
