// Image Merger / Combiner

let mergeImages = [];

window.initializeImageMerger = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="converter-container">
            <div class="converter-header">
                <h2><i class="fas fa-object-group"></i> Image Merger</h2>
                <p>Combine multiple images side by side</p>
            </div>

            <div class="upload-zone" onclick="document.getElementById('merge-input').click()">
                <i class="fas fa-cloud-upload-alt"></i>
                <h3>Drop images here or click to upload</h3>
                <p>Supports: JPG, PNG, WebP (Upload 2-10 images)</p>
                <input type="file" id="merge-input" accept="image/*" multiple style="display: none;">
            </div>

            <div id="merge-preview" class="preview-section hidden">
                <h3>Selected Images (<span id="merge-count">0</span>)</h3>
                <div id="merge-grid" class="image-grid"></div>
            </div>

            <div id="merge-options" class="options-section hidden">
                <h3>Merge Options</h3>
                
                <div class="form-group">
                    <label>Direction:</label>
                    <select id="merge-direction">
                        <option value="horizontal">Horizontal (Side by Side)</option>
                        <option value="vertical">Vertical (Top to Bottom)</option>
                        <option value="grid">Grid (Auto Arrange)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Alignment:</label>
                    <select id="merge-align">
                        <option value="start">Start</option>
                        <option value="center" selected>Center</option>
                        <option value="end">End</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Spacing: <span id="merge-spacing-value">0</span>px</label>
                    <input type="range" id="merge-spacing" min="0" max="50" value="0" oninput="document.getElementById('merge-spacing-value').textContent = this.value">
                </div>

                <div class="form-group">
                    <label>Background Color:</label>
                    <input type="color" id="merge-bg" value="#ffffff">
                </div>

                <button onclick="mergeImagesFunc()" class="btn-primary">
                    <i class="fas fa-object-group"></i> Merge Images
                </button>
            </div>

            <div id="merge-results" class="results-section hidden"></div>
        </div>
    `;

    document.getElementById('merge-input').addEventListener('change', handleMergeUpload);
};

function handleMergeUpload(e) {
    mergeImages = Array.from(e.target.files);
    if (!mergeImages.length) return;

    if (mergeImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }

    if (mergeImages.length > 10) {
        alert('Maximum 10 images allowed');
        mergeImages = mergeImages.slice(0, 10);
    }

    document.getElementById('merge-count').textContent = mergeImages.length;
    document.getElementById('merge-preview').classList.remove('hidden');
    document.getElementById('merge-options').classList.remove('hidden');

    const grid = document.getElementById('merge-grid');
    grid.innerHTML = '';

    mergeImages.forEach((file) => {
        const item = document.createElement('div');
        item.className = 'image-item';
        item.innerHTML = `
            <img src="${URL.createObjectURL(file)}" alt="${file.name}">
            <div class="image-name">${file.name}</div>
        `;
        grid.appendChild(item);
    });
}

async function mergeImagesFunc() {
    if (mergeImages.length < 2) {
        alert('Please select at least 2 images');
        return;
    }

    const resultsDiv = document.getElementById('merge-results');
    resultsDiv.innerHTML = '<div class="processing"><i class="fas fa-spinner fa-spin"></i> Merging images...</div>';
    resultsDiv.classList.remove('hidden');

    const formData = new FormData();
    mergeImages.forEach(file => formData.append('images', file));
    formData.append('direction', document.getElementById('merge-direction').value);
    formData.append('align', document.getElementById('merge-align').value);
    formData.append('spacing', document.getElementById('merge-spacing').value);
    formData.append('backgroundColor', document.getElementById('merge-bg').value);

    try {
        const response = await fetch('/api/image/merge', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            resultsDiv.innerHTML = `
                <div class="results-success">
                    <h3>✅ Images Merged!</h3>
                    <img src="${data.url}" alt="Merged" style="max-width: 100%; border: 1px solid #ddd;">
                    <a href="${data.url}" download="${data.filename}" class="btn-download">
                        <i class="fas fa-download"></i> Download Merged Image
                    </a>
                </div>
            `;
        } else {
            resultsDiv.innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        resultsDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    }
}
