// Image Filters & Effects

let filterImages = [];
let currentFilterImage = null;

window.initializeImageFilters = function() {
    const container = document.getElementById('tool-container');
    container.innerHTML = `
        <div class="filter-interface">
            <div class="upload-section">
                <label class="upload-label">
                    <input type="file" id="filter-images" accept="image/*" multiple onchange="loadImagesForFilters(this)" style="display: none;">
                    <div class="upload-box">
                        <i class="fas fa-images"></i>
                        <p>Select Images</p>
                    </div>
                </label>
            </div>
            <div id="filter-preview" class="preview-section hidden">
                <img id="filter-preview-img" alt="Preview">
            </div>
            <div id="filter-options" class="options-section hidden">
                <h3>Filters & Effects</h3>
                <div class="filter-presets">
                    <button class="filter-preset-btn active" onclick="selectFilter('none')">None</button>
                    <button class="filter-preset-btn" onclick="selectFilter('grayscale')">Grayscale</button>
                    <button class="filter-preset-btn" onclick="selectFilter('sepia')">Sepia</button>
                    <button class="filter-preset-btn" onclick="selectFilter('vintage')">Vintage</button>
                    <button class="filter-preset-btn" onclick="selectFilter('cool')">Cool</button>
                    <button class="filter-preset-btn" onclick="selectFilter('warm')">Warm</button>
                    <button class="filter-preset-btn" onclick="selectFilter('invert')">Invert</button>
                </div>
                <input type="hidden" id="filter-type" value="none">
                <div class="slider-group">
                    <label>Brightness: <span id="brightness-value">100</span>%</label>
                    <input type="range" id="filter-brightness" min="0" max="200" value="100" oninput="applyFilterPreview()">
                </div>
                <div class="slider-group">
                    <label>Contrast: <span id="contrast-value">100</span>%</label>
                    <input type="range" id="filter-contrast" min="0" max="200" value="100" oninput="applyFilterPreview()">
                </div>
                <div class="slider-group">
                    <label>Saturation: <span id="saturation-value">100</span>%</label>
                    <input type="range" id="filter-saturation" min="0" max="200" value="100" oninput="applyFilterPreview()">
                </div>
                <div class="slider-group">
                    <label>Blur: <span id="blur-value">0</span>px</label>
                    <input type="range" id="filter-blur" min="0" max="10" value="0" oninput="applyFilterPreview()">
                </div>
                <div class="form-group">
                    <label>Output Format:</label>
                    <select id="filter-format">
                        <option value="jpg">JPG</option>
                        <option value="png">PNG</option>
                        <option value="webp">WebP</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Quality:</label>
                    <input type="range" id="filter-quality" min="1" max="100" value="90">
                </div>
                <button onclick="applyFilters()" class="btn-primary">Apply to All Images</button>
            </div>
            <div id="filter-results" class="result-section hidden"></div>
        </div>
    `;
};

function loadImagesForFilters(input) {
    filterImages = Array.from(input.files);
    if (!filterImages.length) return;

    currentFilterImage = filterImages[0];
    displayFilterPreview();
    document.getElementById('filter-options').classList.remove('hidden');
}

function displayFilterPreview() {
    const reader = new FileReader();
    reader.onload = function(e) {
        const img = document.getElementById('filter-preview-img');
        img.src = e.target.result;
        img.onload = function() {
            applyFilterPreview();
        };
        document.getElementById('filter-preview').classList.remove('hidden');
    };
    reader.readAsDataURL(currentFilterImage);
}

function selectFilter(filter) {
    document.querySelectorAll('.filter-preset-btn').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    document.getElementById('filter-type').value = filter;
    applyFilterPreview();
}

function applyFilterPreview() {
    const filter = document.getElementById('filter-type').value;
    const brightness = document.getElementById('filter-brightness').value;
    const contrast = document.getElementById('filter-contrast').value;
    const saturation = document.getElementById('filter-saturation').value;
    const blur = document.getElementById('filter-blur').value;
    
    document.getElementById('brightness-value').textContent = brightness;
    document.getElementById('contrast-value').textContent = contrast;
    document.getElementById('saturation-value').textContent = saturation;
    document.getElementById('blur-value').textContent = blur;

    const img = document.getElementById('filter-preview-img');
    
    let filterCSS = `brightness(${brightness}%) contrast(${contrast}%) saturate(${saturation}%) blur(${blur}px)`;
    
    // Add preset filters
    if (filter === 'grayscale') filterCSS += ' grayscale(100%)';
    else if (filter === 'sepia') filterCSS += ' sepia(100%)';
    else if (filter === 'vintage') filterCSS = `brightness(110%) contrast(90%) saturate(120%) sepia(30%)`;
    else if (filter === 'cool') filterCSS += ' hue-rotate(180deg)';
    else if (filter === 'warm') filterCSS += ' hue-rotate(30deg)';
    else if (filter === 'invert') filterCSS += ' invert(100%)';
    
    img.style.filter = filterCSS;
}

async function applyFilters() {
    if (!filterImages.length) return;

    const filter = document.getElementById('filter-type').value;
    const brightness = document.getElementById('filter-brightness').value;
    const contrast = document.getElementById('filter-contrast').value;
    const saturation = document.getElementById('filter-saturation').value;
    const blur = document.getElementById('filter-blur').value;
    const format = document.getElementById('filter-format').value;
    const quality = document.getElementById('filter-quality').value;

    document.getElementById('filter-results').innerHTML = '<div class="processing">Applying filters to images...</div>';
    document.getElementById('filter-results').classList.remove('hidden');

    const formData = new FormData();
    filterImages.forEach(file => formData.append('images', file));
    formData.append('filter', filter);
    formData.append('brightness', brightness);
    formData.append('contrast', contrast);
    formData.append('saturation', saturation);
    formData.append('blur', blur);
    formData.append('format', format);
    formData.append('quality', quality);

    try {
        const response = await fetch('/api/image/apply-filters', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            displayFilterResults(data.images);
        } else {
            document.getElementById('filter-results').innerHTML = `<div class="error">Error: ${data.error}</div>`;
        }
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('filter-results').innerHTML = `<div class="error">Error applying filters: ${error.message}</div>`;
    }
}

function displayFilterResults(images) {
    let html = '<div class="result-success"><h3>✅ Filters Applied Successfully!</h3><div class="filter-grid">';
    
    images.forEach(img => {
        html += `
            <div class="filter-item">
                <img src="${img.url}" alt="${img.filename}">
                <div class="filter-item-info">
                    <p class="filename">${img.filename}</p>
                    <button onclick="window.open('${img.url}', '_blank')" class="download-btn-small">
                        📥 Download
                    </button>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    
    if (images.length > 1) {
        html += '<button onclick="downloadAllFiltered()" class="download-btn">📥 Download All as ZIP</button>';
    }
    
    html += '</div>';
    document.getElementById('filter-results').innerHTML = html;
}

function downloadAllFiltered() {
    alert('Batch download functionality coming soon!');
}
